package com.tenqube.visual_third.ui;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.CardView;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.TextView;

import com.tenqube.visual_third.Callback;
import com.tenqube.visual_third.OnAdLoadedListener;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.analysis.AnalysisHelper;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.manager.AdManager;
import com.tenqube.visual_third.manager.Lv0DataCache;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.SecretKeyManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.js.AdRequest;
import com.tenqube.visual_third.model.js.Lv0Info;
import com.tenqube.visual_third.model.ui.TransactionPopupInfo;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.Utils;
import com.tenqube.visual_third.util.WebViewHelper;
import com.tenqube.visual_third.web.ActionImpl;
import com.tenqube.visual_third.web.ErrorImpl;
import com.tenqube.visual_third.web.LogImpl;
import com.tenqube.visual_third.web.RepoImpl;
import com.tenqube.visual_third.web.SystemImpl;
import com.tenqube.visual_third.web.UiImpl;
import com.tenqube.visual_third.web.VisualInterface;

import java.util.Calendar;

import static android.content.Intent.FLAG_ACTIVITY_REORDER_TO_FRONT;
import static com.tenqube.visual_third.util.WebViewHelper.settings;

/**
 * 서버에서 전달받은 Transaction 객체를 가지고 팝업을 띄운다.
 * 1. intent Transaction 객체
 * 2. validation 체크
 * 3. UI 구성
 */
public class TransactionPopupActivity extends AppCompatActivity implements ActivityCallback {
    public static final String ARG_TRANSACTION = "ARG_TRANSACTION";

    private FrameLayout webContainer;

    private CardView adContainer;
    private TransactionPopupInfo transactionPopupInfo;
    private boolean isPageLoaded;

    private Lv0Info.Lv0 lv0;
    private VisualInterface.Repo repoInterface;
    private VisualInterface.UI uiInterface;
    private VisualInterface.System systemInterface;
    private VisualInterface.Error errorInterface;
    private VisualInterface.Log logInterface;

    private VisualInterface.Action actionInterface;

    private VisualRepository repository;

    public static void start(Context context, TransactionPopupInfo transactionPopupInfo) throws AuthException {
        if(TextUtils.isEmpty(SecretKeyManager.getInstance(context).getKey(PrefManager.ACCESS_TOKEN)))
            throw new AuthException("Need to sign up");


        try {
            Intent intent = new Intent(context, TransactionPopupActivity.class);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                    Intent.FLAG_ACTIVITY_CLEAR_TOP|
                    Intent.FLAG_ACTIVITY_NO_HISTORY|
                    FLAG_ACTIVITY_REORDER_TO_FRONT);

            intent.putExtra(ARG_TRANSACTION, transactionPopupInfo);
            context.startActivity(intent);
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    @SuppressLint({"JavascriptInterface", "AddJavascriptInterface", "SetJavaScriptEnabled"})
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        try {
            transactionPopupInfo = (TransactionPopupInfo) getIntent().getSerializableExtra(ARG_TRANSACTION);

            if(transactionPopupInfo == null ||
                    !RepositoryHolder.getInstance(getApplicationContext()).getVisualRepository().isActiveTranPopup()) {
                finish();
                return;
            }

            onWebViewCreated();
        } catch (Exception e) {
            e.printStackTrace();
            finish();
        }
    }

    private void onWebViewCreated() {

        setContentView(R.layout.visual_activity_transaction_web_popup);
        WebView webView = findViewById(R.id.webview);
        webView.setBackgroundColor(Color.TRANSPARENT);
        webContainer = findViewById(R.id.web_container);
        setupWebView(webView);
//        onAndroidViewCreated();

    }

    private void setupWebView(final WebView webview) {

        repository = RepositoryHolder.getInstance(getApplicationContext()).getVisualRepository();

        // javascriptInterface
        errorInterface = new ErrorImpl(this, webview);
        logInterface = new LogImpl(this, webview, errorInterface);

        repoInterface = new RepoImpl(this, webview, errorInterface,
                repository,
                new AnalysisServiceImpl(this, ""), VisualAlarmManager.getInstance(getApplicationContext()));
        uiInterface = new UiImpl(this, webview, errorInterface, new BottomDialog(this), AdManager.getInstance(getApplicationContext()));
        systemInterface = new SystemImpl(this, webview, errorInterface);

        actionInterface = new ActionImpl(this, webview);

        webview.addJavascriptInterface(errorInterface, "visualError");
        webview.addJavascriptInterface(repoInterface, "visualRepo");
        webview.addJavascriptInterface(uiInterface, "visualUI");
        webview.addJavascriptInterface(systemInterface, "visualSystem");
        webview.addJavascriptInterface(logInterface, "visualLog");
        webview.addJavascriptInterface(actionInterface, "visualAction");

        // settings
        settings(webview, repository.isDev());

        // error
        webview.setWebViewClient(new WebViewClient() {
            public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {

                webview.setVisibility(View.GONE);
                onAndroidViewCreated();
            }
        });

        if(PrefManager.getInstance(getApplicationContext()).loadStringValue(PrefManager.QUALIFIER, "dev").equals("dev"))
            webview.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);


        webview.loadUrl(WebViewHelper.getPopupUrl(PrefManager.getInstance(getApplicationContext())));
    }

    private void onAndroidViewCreated() {

        findViewById(R.id.native_view).setVisibility(View.VISIBLE);
//        setContentView(R.layout.activity_transaction_popup);

        ImageView iconImageView = findViewById(R.id.icon);
        ImageView closeImageView = findViewById(R.id.close);
        closeImageView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                overridePendingTransition(0, 0);

                finish();
            }
        });
        TextView appNameTextView = findViewById(R.id.app_name);
        setAppInfo(iconImageView, appNameTextView);

        TextView keywordTextView = findViewById(R.id.keyword);
        keywordTextView.setText(transactionPopupInfo.getCompanyTitle());

        TextView amountTextView = findViewById(R.id.amount);
        amountTextView.setText(transactionPopupInfo.getSpentMoneyStr());

        TextView addressTextView = findViewById(R.id.address);
        addressTextView.setText(transactionPopupInfo.getCompanyAddress());

        TextView dateTextView = findViewById(R.id.date);
        dateTextView.setText(transactionPopupInfo.getTransaction().getTransaction().getSpentDate());

        TextView cardTextView = findViewById(R.id.card);
        cardTextView.setText(transactionPopupInfo.getPayment());
        TextView lContentTextView = findViewById(R.id.lContent);


        if(Lv0DataCache.getInstance().lv0s == null || Lv0DataCache.getInstance().lv0s.isEmpty()) {
            double sum = repository.getSum(getMonthFrom(), getMonthEnd());
            lContentTextView.setText(String.format("이번 달 지출은\n %1$s 입니다.", AnalysisHelper.getLv0Currency(sum)));
        } else {
            lv0 = Lv0DataCache.getInstance().getLv0(transactionPopupInfo.getTransaction().getCategory().getLarge());
            lContentTextView.setText(lv0.getlContent());
        }

        CardView moreCardView = findViewById(R.id.more);
        moreCardView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                try {
                    String path = "";
                    if(lv0 != null) {
                        path = "&lv0popup=" + Utils.toJson(lv0);
                    }

                    VisualWebActivity.startActivity(getApplicationContext(), path);
                } catch (AuthException e) {
                    e.printStackTrace();
                } finally {
                    finish();
                }

            }
        });

        webContainer = findViewById(R.id.adContainer);

        loadAd();

    }

    private String getMonthFrom() {

        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DATE, 1);
        return Utils.getStringDateAsYYYYMMddHHmmss(calendar);
    }

    private String getMonthEnd() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, 1);

        return Utils.getStringDateAsYYYYMMddHHmmss(calendar);
    }

    private void setAppInfo(ImageView iconImageView, TextView title) {
        PackageManager packageManager = getApplicationContext().getPackageManager();

        try {

            String pkgName = getApplicationContext().getPackageName();
            ApplicationInfo appInfo = packageManager.getApplicationInfo(pkgName, PackageManager.GET_META_DATA);
            Drawable icon = packageManager.getApplicationIcon(pkgName);
            iconImageView.setImageDrawable(icon);
            String appName = (String) packageManager.getApplicationLabel(appInfo);
            title.setText(appName);
        } catch (PackageManager.NameNotFoundException e) {

        }
    }


    private void loadAd() {
        AdRequest adRequest = new AdRequest();
        adRequest.unitId = "ca-app-pub-1003660361092577/2414077097";
        AdManager.getInstance(getApplicationContext()).loadAd(adRequest, new OnAdLoadedListener() {
            @Override
            public void onAdLoaded(View view) {

                try {
                    webContainer.addView(createCardView(view));
                } catch (Exception e) {
                    e.printStackTrace();
                }

            }
        });
    }

    @Override
    public void exportExcel(Callback<Boolean> callback) {

    }

    @Override
    public void onBackPressed() {

        try {
            if(isPageLoaded && uiInterface != null) {
                uiInterface.onFinish();
            } else {
                super.onBackPressed();
            }
        } catch (Exception e) {
            super.onBackPressed();
        }
    }

    @Override
    public Context getContext() {
        return getApplicationContext();
    }

    @Override
    public AppCompatActivity getActivity() {
        return this;
    }

    @Override
    public void onPagedLoaded() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                isPageLoaded = true;
            }
        });
    }

    @Override
    public void onScrollChanged(int pos) {

    }

    public void reload() {

    }

    public void retry() {

    }

    @Override
    public void setRefreshEnabled(boolean enabled) {

    }

    private CardView createCardView(View view) {
        adContainer = new CardView(this);
        adContainer.setCardElevation(getResources().getDimension(R.dimen.elevation));
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            adContainer.setElevation(getResources().getDimension(R.dimen.elevation));
        }
        // Set the CardView layoutParams
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.WRAP_CONTENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
        );

        params.setMargins(Utils.dpToPx(10), Utils.dpToPx(16), Utils.dpToPx(10), Utils.dpToPx(72));
        params.gravity = Gravity.BOTTOM;
        adContainer.setLayoutParams(params);

        // Set CardView corner radius
        adContainer.setRadius(Utils.dpToPx(13));

        // Set a background color for CardView
        adContainer.setCardBackgroundColor(Color.parseColor("#00000000"));


        adContainer.addView(view);

        return adContainer;

    }

    public void addAd(final View view) {
        try {
            webContainer.addView(createCardView(view));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void hideAd() {
        try {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if(adContainer != null) {
                        webContainer.removeView(adContainer);
                    }

                }
            });

        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Override
    public void finish() {
        super.finish();
        overridePendingTransition(R.anim.stay, 0);
    }

    @Override
    public void onResume() {
        super.onResume();
        overridePendingTransition(0, 0);
    }

    @Override
    public void onPause() {
        super.onPause();
        overridePendingTransition(0, 0);
    }

    @Override
    public TransactionPopupInfo getPopUpInfo() {
        return transactionPopupInfo;
    }

    @Override
    public void setStatusBarColor(String color) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            try {
                getWindow().setStatusBarColor(Color.parseColor(color));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }


}