package com.tenqube.visual_third.api.auth;

import android.content.Context;

import com.tenqube.visual_third.api.data.ResultBody;
import com.tenqube.visual_third.api.data.Token;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.SecretKeyManager;
import com.tenqube.visual_third.model.api.UserRequest;
import com.tenqube.visual_third.model.api.UserResult;
import com.tenqube.visual_third.util.Validator;

import java.io.IOException;
import java.util.Calendar;

import retrofit2.Response;

public class AuthServiceImpl implements AuthService {

    private static final String TAG = AuthServiceImpl.class.getSimpleName();

    private final AuthApi authApi;
    private final PrefManager prefManager;
    private final SecretKeyManager secretKeyManager;
    private final Context context;

    private AuthServiceImpl(Context context, AuthApi authApi, PrefManager prefManager, SecretKeyManager secretKeyManager) {
        this.context = context;
        this.authApi = authApi;
        this.prefManager = prefManager;
        this.secretKeyManager = secretKeyManager;
    }

    private static AuthService mInstance;

    public static AuthService getInstance(Context context, AuthApi authApi, PrefManager prefManager, SecretKeyManager secretKeyManager) {
        synchronized (AuthService.class) {
            if (mInstance == null) {
                mInstance = new AuthServiceImpl(context, authApi, prefManager, secretKeyManager);
            }
        }
        return mInstance;
    }

    @Override
    public boolean signUp(UserRequest request) throws IOException {
        Response<ResultBody<UserResult>> response;

        response = authApi.signUp(request).execute();

        if (response.isSuccessful()) {

            ResultBody<UserResult> result = response.body();
            if (result != null && result.results != null) {
                saveUserInfo(request, result.results);
                return true;
            }
        }
        return false;

    }

    @Override
    public boolean reissueToken() throws Exception {

        String refreshToken = secretKeyManager.getKey(PrefManager.REFRESH_TOKEN);

        Validator.isNotEmpty(refreshToken);

        retrofit2.Response<ResultBody<Token>> result = authApi.reissueToken(refreshToken).execute();
        if (result.isSuccessful()) {
            ResultBody<Token> results = result.body();
            if(results != null && results.results != null) {
                saveToken(results.results);
                return true;
            }
        } else {
            int statusCode = result.code();
            if(statusCode == 403) {
                signOut();
            } else if(statusCode == 401) {
                UserRequest userRequest = new UserRequest(prefManager.loadStringValue(PrefManager.USER_ID, ""),
                        prefManager.loadStringValue(PrefManager.USER_AD_ID, ""));
                boolean success = signUp(userRequest);
                if(success) {
                    return true;
                }
            }
        }

        return false;
    }

    @Override
    public String getAccessToken() {
        // 401 test token
      return secretKeyManager.getKey(PrefManager.ACCESS_TOKEN);
    }

    private void saveUserInfo(UserRequest user, UserResult response) {

        if(user != null) {
            prefManager.saveStringValue(PrefManager.USER_ID, user.getUid());
            prefManager.saveStringValue(PrefManager.USER_AD_ID, user.getAdId());
        }

        tenqube.parser.core.SecretKeyManager.getInstance(context).saveKey(response.getSecretKey());

        saveToken(response.getAuthorization().getSdk());

        prefManager.saveStringValue(PrefManager.RESOURCE_URL, response.getResource().getUrl());
        prefManager.saveStringValue(PrefManager.RESOURCE_API_KEY, response.getResource().getApiKey());

        prefManager.saveStringValue(PrefManager.SEARCH_URL, response.getSearch().getUrl());
        prefManager.saveStringValue(PrefManager.SEARCH_API_KEY, response.getSearch().getApiKey());

        prefManager.saveLongValue(PrefManager.SIGN_UP_TIME, Calendar.getInstance().getTimeInMillis());
        if (response.getWeb() != null)
            prefManager.saveStringValue(PrefManager.WEB_URL, response.getWeb().getUrl());

    }

    private void saveToken(Token token) {
        if(token == null) return;
        secretKeyManager.save(PrefManager.REFRESH_TOKEN, token.getAccess());
        secretKeyManager.save(PrefManager.ACCESS_TOKEN, token.getRefresh());
    }

    @Override
    public void signOut() {
        prefManager.saveStringValue(PrefManager.REFRESH_TOKEN, "");
        prefManager.saveStringValue(PrefManager.ACCESS_TOKEN, "");

    }
}
