package com.tenqube.visual_scraper.crawler.manager;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.webkit.JsResult;
import android.webkit.WebChromeClient;
import android.webkit.WebResourceError;
import android.webkit.WebResourceRequest;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.crawler.ScrapInterface;

import java.util.Timer;
import java.util.TimerTask;

import timber.log.Timber;

public class WebManager {

    private Handler timerHandler = new Handler();

    private Timer timer;

    private TimerTask timerTask;

    private Context context;

    private WebView webView;

    private OnWebInteractionListener mListener;

    public interface OnWebInteractionListener {

        int getTimeOut();

        void onTimeOut();

        void showHtml(String html);

        void checkHtml(String html);

        void onReceivedError(String msg);

        void onPageFinished(WebView view, String url);

    }

    @SuppressLint({"JavascriptInterface",
            "SetJavaScriptEnabled",
            "AddJavascriptInterface"})
    public WebManager(Context context, OnWebInteractionListener listener) {
        this.context = context;
        this.mListener = listener;
        initWebView();
    }

    @SuppressLint({"SetJavaScriptEnabled",
            "AddJavascriptInterface"})
    private void initWebView() {
        this.webView = new WebView(context);
        webView.getSettings().setJavaScriptEnabled(true);
        webView.getSettings().setUserAgentString(Constants.USER_AGENT_NORMAL);
        webView.addJavascriptInterface(new ScrapInterface(html -> mListener.showHtml(html)), "HtmlViewer");

        webView.getSettings().setDomStorageEnabled(true);
        if (Build.VERSION.SDK_INT >= 21) webView.getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_COMPATIBILITY_MODE);
        webView.setLayerType(WebView.LAYER_TYPE_SOFTWARE, null);
        webView.setWebViewClient(new WebViewClient() {
            @Override
            public void onReceivedError(WebView view, WebResourceRequest request, WebResourceError error) {
                super.onReceivedError(view, request, error);
                stopTimer();
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    mListener.onReceivedError(error.getDescription().toString());
                } else {
                    mListener.onReceivedError(error.toString());
                }
            }

            @Override
            public void onPageFinished(WebView view, String url) {
                super.onPageFinished(view, url);
                stopTimer();
                if ("about:blank".equals(url) && view.getTag() != null) {
                    loadUrl(view.getTag().toString());
                    return;
                }
                view.setTag(url);

                mListener.onPageFinished(view, url);
            }
        });

        webView.setWebChromeClient(new WebChromeClient(){
            @Override
            public boolean onJsAlert(WebView view, String url, String message, JsResult result) {
//                LoginFailed(message);
                result.confirm();
                return true;
            }
        });
    }

    @SuppressLint("SetJavaScriptEnabled")
    public void loadUrl(String urlOrScripts) {
        Timber.i("loadWebView"+"url :"+ urlOrScripts);
        startTimer();
        webView.loadUrl(urlOrScripts, null);
    }

    public WebView getWebView(){
        return webView;
    }

    private void startTimer() {
        timer = new Timer();
        timerTask = new TimerTask() {
            @Override
            public void run() {

                timerHandler.post(() -> mListener.onTimeOut());
            }
        };
        timer.schedule(timerTask, mListener.getTimeOut());
    }

    private void stopTimer(){
        if(timer != null){
            timer.cancel();
            timer.purge();
            timer = null;
            timerTask = null;
        }
    }
}
