package com.tenqube.visual_scraper.repository;

import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;
import android.util.SparseArray;

import com.tenqube.visual_scraper.ScrapService;
import com.tenqube.visual_scraper.api.ScrapApiService;
import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.db.ScraperDatabase;
import com.tenqube.visual_scraper.db.dao.LoginApiRuleDao;
import com.tenqube.visual_scraper.db.dao.LoginRuleDao;
import com.tenqube.visual_scraper.db.dao.LoginWebRuleDao;
import com.tenqube.visual_scraper.db.dao.MallDao;
import com.tenqube.visual_scraper.db.dao.MallWithUserDao;
import com.tenqube.visual_scraper.db.dao.OrderApiRuleDao;
import com.tenqube.visual_scraper.db.dao.OrderDao;
import com.tenqube.visual_scraper.db.dao.OrderDetailWebRuleDao;
import com.tenqube.visual_scraper.db.dao.OrderWebRuleDao;
import com.tenqube.visual_scraper.db.dao.UserDao;
import com.tenqube.visual_scraper.db.dao.UserWithOrdersDao;
import com.tenqube.visual_scraper.db.entity.LoginWebRule;
import com.tenqube.visual_scraper.db.entity.MallEntity;
import com.tenqube.visual_scraper.db.entity.OrderApiRuleJson;
import com.tenqube.visual_scraper.db.entity.OrderDetailWebRuleJson;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.db.entity.OrderWebRule;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.model.api.request.OrderBody;
import com.tenqube.visual_scraper.model.api.request.SaveOrdersBody;
import com.tenqube.visual_scraper.model.api.request.SignUpBody;
import com.tenqube.visual_scraper.model.api.response.LoginWebRuleResponse;
import com.tenqube.visual_scraper.model.api.response.MallResponse;
import com.tenqube.visual_scraper.model.api.response.OrderDetailHtmlResponse;
import com.tenqube.visual_scraper.model.api.response.OrderWebRuleResponse;
import com.tenqube.visual_scraper.model.api.response.ResultBody;
import com.tenqube.visual_scraper.model.api.response.ResultRuleVersion;
import com.tenqube.visual_scraper.model.api.response.ResultSignUp;
import com.tenqube.visual_scraper.model.api.response.RuleResponse;
import com.tenqube.visual_scraper.model.query.LoginRule;
import com.tenqube.visual_scraper.model.query.MallWithUser;
import com.tenqube.visual_scraper.model.view.MallInfo;
import com.tenqube.visual_scraper.model.view.OrderInfo;
import com.tenqube.visual_scraper.model.view.ViewMoreInfo;
import com.tenqube.visual_scraper.utils.AppExecutors;
import com.tenqube.visual_scraper.utils.DateUtils;
import com.tenqube.visual_scraper.utils.Utils;

import java.util.ArrayList;
import java.util.List;

import retrofit2.Response;
import timber.log.Timber;

public class ScrapRepository {

    private static ScrapRepository sInstance;

    private AppExecutors appExecutors;
    private final ScraperDatabase mDatabase;
    private final UserDao userDao;
    private final OrderDao orderDao;
    private final UserWithOrdersDao userWithOrdersDao;
    private final MallWithUserDao mallWithUserDao;
    private final MallDao mallDao;
    private final LoginRuleDao loginRuleDao;

    private final LoginApiRuleDao loginApiRuleDao;
    private final LoginWebRuleDao loginWebRuleDao;
    private final OrderApiRuleDao orderApiRuleDao;
    private final OrderWebRuleDao orderWebRuleDao;
    private final OrderDetailWebRuleDao orderDetailWebRuleDao;

    private final ScrapApiService mApi;
    private SharedPreferences preferences;

    private ScrapRepository(final ScraperDatabase database,
                            final ScrapApiService scrapApi,
                            final AppExecutors appExecutors,
                            final SharedPreferences preferences) {

        // local
        mDatabase = database;
        userDao = database.userDao();
        orderDao = database.orderDao();
        userWithOrdersDao = database.userWithOrdersDao();
        mallWithUserDao = database.mallWithUserDao();
        mallDao = database.mallDao();
        loginRuleDao = database.loginRuleDao();
        loginApiRuleDao = database.loginApiRuleDao();
        loginWebRuleDao = database.loginWebRuleDao();
        orderApiRuleDao = database.orderApiRuleDao();
        orderWebRuleDao = database.orderWebRuleDao();
        orderDetailWebRuleDao = database.orderDetailWebRuleDao();
        this.preferences = preferences;

        // remote
        mApi = scrapApi;
        this.appExecutors = appExecutors;
    }

    public static ScrapRepository getInstance(final ScraperDatabase database, final ScrapApiService scrapApi, final AppExecutors appExecutors, final SharedPreferences preferences) {
        if (sInstance == null) {
            synchronized (ScrapRepository.class) {
                if (sInstance == null) {
                    sInstance = new ScrapRepository(database, scrapApi, appExecutors, preferences);
                }
            }
        }
        return sInstance;
    }

    public void saveKey(String apiKey) {
        if (TextUtils.isEmpty(apiKey)) apiKey = "sVPnrNW0Fg4fujRSLJrYn8FJWyO5BtS21hAbOYXc"; //TEST ***
        preferences.edit().putString(Constants.PREF_KEY.X_API_KEY, apiKey).apply();
    }

    public void saveLayer(String layer) {
        preferences.edit().putString(Constants.PREF_KEY.LAYER, layer).apply();
    }

    public void signUpUser(String appName, String uuid, String adId, ScrapService.Callback<Boolean> callback) {
        appExecutors.networkIO().execute(() -> {

            Response<ResultBody<ResultSignUp>> response;
            try {
                response = mApi.callSignUpUser(new SignUpBody(appName, uuid, adId)).execute();

                if (response.isSuccessful()) {
                    Timber.i("signUpUser" +"success : "+response.body());
                    preferences.edit().putString(Constants.PREF_KEY.UUID, uuid).apply();
                    ResultBody<ResultSignUp> result = response.body();
                    if (result != null) {
                        preferences.edit().putString(Constants.PREF_KEY.TOKEN, result.result.authorization.mall).apply();
                        preferences.edit().putString(Constants.PREF_KEY.SECRET_KEY, result.result.secretKey).apply();
                        appExecutors.mainThread().execute(() -> callback.onDataLoaded(true));
                    }

                } else {
                    Timber.i("signUpUser" +"error :"+response.message());
                    appExecutors.mainThread().execute(() -> callback.onDataLoaded(false));
                }

            } catch (Exception e) {
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(false));
            }
        });
    }

    public void syncRule(@Nullable ScrapService.onResultInterface<Void> callback) {

        appExecutors.networkIO().execute(() -> {
            try {
                Response<ResultRuleVersion> versionResponse = mApi.callGetRuleVersion().execute();
                if (versionResponse.isSuccessful()) {
                    ResultRuleVersion serverVersion = versionResponse.body();

                    if (serverVersion != null) {
                        int clientVersion =  preferences.getInt(Constants.PREF_KEY.RULE_VERSION, -1);

                        if(serverVersion.version > clientVersion) {
                            callRuleAndInsertDB();

                        }// check version end
                    }// res null check end
                }// version suc end

                appExecutors.mainThread().execute(()-> {
                    if (callback != null) {
                        callback.onDataLoaded(null);
                    }
                });

            } catch (Exception e) {
                e.printStackTrace();
                appExecutors.mainThread().execute(()-> {
                    if (callback != null) {
                        callback.onDataLoaded(null);
                    }
                });
            }
        });
    }

    private void callRuleAndInsertDB() throws Exception {
        Response<ResultBody<RuleResponse>> response = mApi.callGetRules().execute();

        if (response.isSuccessful()) {

            // json 파싱 후 저장
            ResultBody<RuleResponse> results = response.body();

            if (results != null) {

                if (results.result.getMalls() != null) {
                    insertMalls(results.result.getMalls());
                }

                if (results.result.getLoginHtmlRules() != null) {
                    insertLoginWebRules(results.result.getLoginHtmlRules());
                }

                if (results.result.getOrderHtmlRules() != null) {
                    insertOrderWebRules(results.result.getOrderHtmlRules());
                }

                if (results.result.getOrderDetailHtmlRules() != null) {
                    insertOrderDetailWebRules(results.result.getOrderDetailHtmlRules());
                }

                // Save new version data
                preferences.edit().putInt(Constants.PREF_KEY.RULE_VERSION, results.result.getVersion()).apply();

            } //body end
        }// rule suc end
    }

    private void insertMalls(List<MallResponse> results) {

        List<MallEntity> malls = new ArrayList<>();
        for(MallResponse res : results) {
            malls.add(res.toEntity());
        }
        mallDao.insertList(malls);
    }

    private void insertLoginWebRules(List<LoginWebRuleResponse> results) {
        ArrayList<LoginWebRule> values = new ArrayList<>();
        for(LoginWebRuleResponse res : results) {
            values.add(res.toEntity());
        }
        List<Long> ids = loginWebRuleDao.insertList(values);

        Timber.i("ids" + ids.toString());

    }

    private void insertOrderWebRules(List<OrderWebRuleResponse> results) {
        ArrayList<OrderWebRule> values = new ArrayList<>();
        for(OrderWebRuleResponse res : results) {
            values.add(res.toEntity());
        }
        orderWebRuleDao.insertList(values);
    }

    private void insertOrderDetailWebRules(List<OrderDetailHtmlResponse> results) {
        ArrayList<OrderDetailWebRuleJson> values = new ArrayList<>();
        for(OrderDetailHtmlResponse res : results) {
            values.add(res.toEntity());
        }
        orderDetailWebRuleDao.insertList(values);
    }

    public void callInsertOrderList() {

        appExecutors.networkIO().execute(() -> {

            if(!TextUtils.isEmpty(preferences.getString(Constants.PREF_KEY.TOKEN, ""))) {
                // 서버에 저장되어있는지 체크
                List<OrderEntity> dataList = orderDao.getOrdersForServer();

                if (dataList.size() > 0) {

                    List<OrderBody> results = new ArrayList<>();
                    List<Integer> ids = new ArrayList<>();
                    for (OrderEntity o: dataList) {
                        OrderBody orderBody = new OrderBody(o.mId,
                                o.title,
                                DateUtils.transformDateStr(o.orderDate),
                                o.orderHms == null ? "" : DateUtils.transformHmsStr(o.orderHms),
                                o.orderOption,
                                o.quantity,
                                o.price,
                                o.currency,
                                o.orderNum,
                                o.orderState,
                                o.imgUrl);

                        results.add(orderBody);
                        ids.add(o._id);
                        Timber.i("saveOrderBody"+"info : "+orderBody);
                    }

                    try {
                        Timber.i("result"+"SaveOrdersBody : "+new SaveOrdersBody(results));
                        Response<String> response = mApi.callSaveOrders(new SaveOrdersBody(results)).execute();
                        if (response.isSuccessful()) {
                            String result = response.body();
                            orderDao.updateIsSendServer(ids);
                            Timber.i("callInsertOrderList"+"result : "+result);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }

        });

    }

    public void getOrderList(ScrapService.Callback<List<OrderEntity>> callback, int... mIds) {
        appExecutors.diskIO().execute(() -> {
            List<OrderEntity> list = orderDao.getOrders(mIds);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(list));
        });
    }

    // send api
    public void getOrderList(ScrapService.Callback<List<OrderEntity>> callback, int mId) {
        appExecutors.diskIO().execute(() -> {
            List<OrderEntity> list = orderDao.getOrders(mId);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(list));
        });
    }

    // send view
    public void getOrderList(int limit, ViewMoreInfo viewMoreInfo, ScrapService.Callback<List<OrderInfo>> callback) {
        appExecutors.diskIO().execute(() -> {

            boolean isMore = viewMoreInfo != null;

            List<OrderEntity> list = isMore ?
                    getMoreOrderList(limit, viewMoreInfo)
                    :
                    orderDao.getOrderList(limit);


            ArrayList<Integer> mIds = new ArrayList<>();
            for(OrderEntity order : list) {
                mIds.add(order.getmId());
            }

            List<MallEntity> malls = mallDao.findMallByIds(mIds);

            SparseArray<MallEntity> mallSparseArray = new SparseArray<>();

            for(MallEntity mallEntity : malls) {
                mallSparseArray.append(mallEntity.getId(), mallEntity);
            }

            appExecutors.mainThread().execute(() -> callback.onDataLoaded(Utils.convertOrderInfo(list, mallSparseArray)));
        });

    }

    private List<OrderEntity> getMoreOrderList(int limit, @NonNull ViewMoreInfo viewMoreInfo) {

            int limitOrderId = viewMoreInfo.getLastOrderId() == 0 ?
                    orderDao.getLastOrderId()
                    :
                    viewMoreInfo.getLastOrderId();

            return orderDao.getOrdersForViewMore(limit, limitOrderId, viewMoreInfo.getBottomOrderDate(),
                    viewMoreInfo.getBottomOrderId());

    }

    public void updateUserLastScrapAt(long lastScrapAt, int mallId, String userId){
        try {

            appExecutors.diskIO().execute(() -> userDao.updateUserLastScrapAt(lastScrapAt, mallId, userId));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void updateUserForSignOut(int mId, ScrapService.OnResultCallback<Void> callback) {
        try {

            appExecutors.diskIO().execute(() -> {
                long i = userDao.updateUserIsLogin(mId, false);
                Timber.i("updateUserForSignOut"+"i : "+i);
                appExecutors.mainThread().execute(new Runnable() {
                    @Override
                    public void run() {
                        if (i > 0 ) {
                            callback.onDataLoaded(null);
                        } else {
                            callback.onFail(mId, Constants.ERROR.LOGIN_OUT_FAIL, "logout fail");
                        }
                    }
                });

            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public void insertOrders(List<OrderEntity> orderEntityList, ScrapService.Callback<List<OrderEntity>> callback) {
        try {
            appExecutors.diskIO().execute(() -> {
                List<Long> savedIds = orderDao.insertList(orderEntityList);
                Timber.i("insertOrders"+"savedIds size : "+savedIds.size() +" / "+savedIds);
                List<Long> resultIds = new ArrayList<>();
                for(Long l : savedIds) {
                    if (l != -1) resultIds.add(l);
                }
                Timber.i("insertOrders" +"resultIds size : "+resultIds.size() +" / "+resultIds);

                // 실제 저장된것만 리스트화
                List<OrderEntity> savedList = orderDao.getOrderList(resultIds);
                Timber.i("insertOrders" +"savedList size : "+savedList.size() +" / "+savedList);

                appExecutors.mainThread().execute(()-> {
                    callback.onDataLoaded(savedList);
                });
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void insertUsers(UserEntity userEntity) {
        try {
            appExecutors.diskIO().execute(() -> userDao.insert(userEntity));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void hideOrder(int orderId) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.hideOrder(orderId));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteOrder(int orderId) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteOrder(orderId));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteOrders(int... orderIds) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteOrders(orderIds));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteOrderByMallIds(int... mIds) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteOrderForMallIds(mIds));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteAllOrders() {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteAll());

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void initialize() {

        appExecutors.diskIO().execute(() -> {
            preferences.edit().clear().apply();
            mDatabase.clearAllTables();
        });
    }

//    public void getUser(String userId, String pwd, Constants.MALL mall, ScrapService.Callback<UserEntity> callback) {
//
//        Runnable runnable = () -> {
//
//            UserEntity userEntity = null;
//            try {
//                userEntity = userDao.getUser(userId, pwd, mall.name());
//
//                if(userEntity == null) {
//                    userEntity = new UserEntity(0, mall.name(), userId, pwd, "");
//                    userEntity.setId((int) userDao.insertList(userEntity));
//                }
//            } catch (Exception e) {
//                e.printStackTrace();
//            } finally {
//                UserEntity finalUserEntity = userEntity;
//                appExecutors.mainThread().execute(() -> callback.onDataLoaded(finalUserEntity));
//            }
//        };
//
//        appExecutors.diskIO().execute(runnable);
//
//    }

//    public void getUser(Constants.MALL mall, ScrapService.Callback<UserEntity> callback) {
//
//        Runnable runnable = () -> {
//
//            UserEntity userEntity = null;
//            try {
//                userEntity = userDao.getUser(mall.name());
//            } catch (Exception e) {
//                e.printStackTrace();
//            } finally {
//                UserEntity finalUserEntity = userEntity;
//                appExecutors.mainThread().execute(() -> callback.onDataLoaded(finalUserEntity));
//            }
//        };
//
//        appExecutors.diskIO().execute(runnable);
//
//    }


//
//    public void deleteUser(UserEntity userEntity) {
//        Runnable runnable = () -> {
//            try {
//                userDao.deleteUser(userEntity.getId());
//            } catch (Exception e) {
//                e.printStackTrace();
//            }
//        };
//
//        appExecutors.diskIO().execute(runnable);
//    }

    public void checkIsActiveMall(int mId, ScrapService.Callback<MallEntity> callback) {
        Runnable runnable = () -> {
            MallEntity mallEntity = mallDao.getMall(mId);
            appExecutors.mainThread().execute(() -> {
                callback.onDataLoaded(mallEntity);
            });
        };
        appExecutors.diskIO().execute(runnable);
    }

    public void getMallList(ScrapService.Callback<List<MallInfo>> callback) {
        appExecutors.diskIO().execute(() -> {
            try {
                List<MallEntity> list = mallDao.getMalls();
                List<MallInfo> results = new ArrayList<>();

                for (MallEntity mallEntity : list) {
                    results.add(new MallInfo(mallEntity.getId(),
                            mallEntity.getDisplayName(),
                            mallEntity.getIconUrl()));
                }
                appExecutors.mainThread().execute(() -> {
                    callback.onDataLoaded(results);
                });

            } catch (Exception e) {
                e.printStackTrace();
            }
        });
    }

    public void getMallWithUser(int mId, String uId, ScrapService.onResultInterface<MallWithUser> callback) {
        appExecutors.diskIO().execute(() -> {
            try {
                MallWithUser mallWithUser = mallWithUserDao.getInfo(mId, uId);
                Log.i("getMallWithUser","info : "+mallWithUser);

                appExecutors.mainThread().execute(() -> {
                    if (mallWithUser != null) {
                        callback.onDataLoaded(mallWithUser);
                    } else {
                        callback.onFail();
                    }
                });

            } catch (Exception e) {
                e.printStackTrace();
            }
        });

    }

    public void getMallWithUser(ScrapService.Callback<List<MallWithUser>> callback) {

        appExecutors.diskIO().execute(() -> {
            try {

                List<MallEntity> malls = mallDao.getMalls();
                List<UserEntity> users = userDao.findUsersByLoginState(1);

                SparseArray<UserEntity> userEntitySparseArray = new SparseArray<>();

                for(UserEntity user: users) {
                    userEntitySparseArray.put(user.getMallId(), user);
                }

                List<MallWithUser> list = new ArrayList<>();

                for(MallEntity mall : malls) {

                    list.add(new MallWithUser(mall, userEntitySparseArray.get(mall.getId())));
                }

                appExecutors.mainThread().execute(() -> {
                    callback.onDataLoaded(list);
                });

            } catch (Exception e) {
                e.printStackTrace();
            }
        });

    }

    public void getMallWithActiveUsers(ScrapService.Callback<List<MallWithUser>> callback) {

        appExecutors.diskIO().execute(() -> {
            try {
                List<MallWithUser> list = mallWithUserDao.getInfosForActiveUser();
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(list));

            } catch (Exception e) {
                e.printStackTrace();
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(new ArrayList<>()));
            }
        });

    }

    public void getMallWithActiveUser(int mallId, ScrapService.Callback<MallWithUser> callback) {

        appExecutors.diskIO().execute(() -> {
            try {
                MallWithUser mallWithUser = mallWithUserDao.getInfoForActiveUser(mallId);
                Log.i("getMallWithActiveUser", "info : "+ mallWithUser);
                appExecutors.mainThread().execute(() -> {
                    callback.onDataLoaded(mallWithUser);
                });

            } catch (Exception e) {
                e.printStackTrace();
            }
        });

    }


    public void getLoginRule(int mId, ScrapService.Callback<LoginRule> callback) {
        appExecutors.diskIO().execute(() -> {
            LoginWebRule loginWebRule = loginWebRuleDao.findById(mId);
            LoginRule loginRule = new LoginRule();
            loginRule.loginWebRule = loginWebRule;
            Log.i("login","loginRule : "+loginRule);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(loginRule));
        });
    }


    public void getOrderWebRule(int mId, ScrapService.Callback<OrderWebRule> callback) {
        appExecutors.diskIO().execute(() -> {
            OrderWebRule orderWebRuleJson = orderWebRuleDao.getInfo(mId);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(orderWebRuleJson));
        });
    }

    public void getOrderApiRule(int mId, ScrapService.Callback<OrderApiRuleJson> callback) {
        appExecutors.diskIO().execute(() -> {
            OrderApiRuleJson orderApiRuleJson = orderApiRuleDao.getInfo(mId);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(orderApiRuleJson));
        });
    }

    public void getDetailHtmlRule(int mId, ScrapService.Callback<OrderDetailWebRuleJson> callback) {
        appExecutors.diskIO().execute(() -> {

            OrderDetailWebRuleJson orderDetailWebRuleJson = orderDetailWebRuleDao.getInfo(mId);
            appExecutors.mainThread().execute(()-> callback.onDataLoaded(orderDetailWebRuleJson));
        });

    }

    public void getLastOrderId(ScrapService.Callback<Integer> callback) {
        appExecutors.diskIO().execute(() -> {

            int lastId = orderDao.getLastOrderId();
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(lastId));
        });
    }

    public void deleteUsersByMallId(int mallId) {
        appExecutors.diskIO().execute(() -> {
            try {
                userDao.deleteUsersByMallId(mallId);
            } catch (Exception e) {
                e.printStackTrace();
            }
        });
    }
}
