/*
 * Copyright 2017, The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.tenqube.visual_scraper.db;

import android.arch.lifecycle.LiveData;
import android.arch.lifecycle.MutableLiveData;
import android.arch.persistence.db.SupportSQLiteDatabase;
import android.arch.persistence.room.Database;
import android.arch.persistence.room.Room;
import android.arch.persistence.room.RoomDatabase;
import android.arch.persistence.room.TypeConverters;
import android.content.Context;
import android.support.annotation.NonNull;
import android.util.Log;

import com.tenqube.visual_scraper.db.dao.LoginApiRuleDao;
import com.tenqube.visual_scraper.db.dao.LoginRuleDao;
import com.tenqube.visual_scraper.db.dao.LoginWebRuleDao;
import com.tenqube.visual_scraper.db.dao.MallDao;
import com.tenqube.visual_scraper.db.dao.MallWithUserDao;
import com.tenqube.visual_scraper.db.dao.OrderApiRuleDao;
import com.tenqube.visual_scraper.db.dao.OrderDao;
import com.tenqube.visual_scraper.db.dao.OrderDetailWebRuleDao;
import com.tenqube.visual_scraper.db.dao.OrderWebRuleDao;
import com.tenqube.visual_scraper.db.dao.UserDao;
import com.tenqube.visual_scraper.db.dao.UserWithOrdersDao;
import com.tenqube.visual_scraper.db.entity.LoginApiRule;
import com.tenqube.visual_scraper.db.entity.LoginWebRule;
import com.tenqube.visual_scraper.db.entity.MallEntity;
import com.tenqube.visual_scraper.db.entity.OrderApiRuleJson;
import com.tenqube.visual_scraper.db.entity.OrderDetailWebRuleJson;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.db.entity.OrderWebRuleJson;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.model.jsonParsing.MallScrapInfo;
import com.tenqube.visual_scraper.utils.AppExecutors;
import com.tenqube.visual_scraper.utils.Utils;

/**
 * The Room Database that contains the ScreenImage table.
 */
@Database(entities = {
        UserEntity.class, OrderEntity.class, MallEntity.class,
        LoginApiRule.class, LoginWebRule.class, OrderApiRuleJson.class, OrderWebRuleJson.class, OrderDetailWebRuleJson.class},
        version = 1, exportSchema = false)
@TypeConverters({Converters.class})
public abstract class ScraperDatabase extends RoomDatabase {

    public static final String DATABASE_NAME = "online_malls";

    private static ScraperDatabase sInstance;

    public abstract UserDao userDao();

    public abstract OrderDao orderDao();

    public abstract UserWithOrdersDao userWithOrdersDao();

    public abstract MallWithUserDao mallWithUserDao();

    public abstract MallDao mallDao();

    public abstract LoginRuleDao loginRuleDao();

    public abstract LoginWebRuleDao loginWebRuleDao();

    public abstract LoginApiRuleDao loginApiRuleDao();

    public abstract OrderApiRuleDao orderApiRuleDao();

    public abstract OrderWebRuleDao orderWebRuleDao();

    public abstract OrderDetailWebRuleDao orderDetailWebRuleDao();




    private final MutableLiveData<Boolean> mIsDatabaseCreated = new MutableLiveData<>();


    public static ScraperDatabase getInstance(final Context context, final AppExecutors executors) {
        if (sInstance == null) {
            synchronized (ScraperDatabase.class) {
                if (sInstance == null) {
                    sInstance = buildDatabase(context.getApplicationContext(), executors);
                    sInstance.updateDatabaseCreated(context.getApplicationContext());
                }
            }
        }
        return sInstance;
    }
    /**
     * Build the database. {@link Builder#build()} only sets up the database configuration and
     * creates a new instance of the database.
     * The SQLite database is only created when it's accessed for the first time.
     */
    private static ScraperDatabase buildDatabase(final Context appContext,
                                             final AppExecutors executors) {

        return Room.databaseBuilder(appContext, ScraperDatabase.class, DATABASE_NAME)
                .addCallback(new Callback() {
                    @Override
                    public void onCreate(@NonNull SupportSQLiteDatabase db) {
                        super.onCreate(db);
                        executors.diskIO().execute(() -> {
                            // Add a delay to simulate a long-running operation
                            addDelay();
                            // Generate the data for pre-population
                            ScraperDatabase database = ScraperDatabase.getInstance(appContext, executors);
//                            List<ScrapEntity> scrapEntities = DataGenerator.generateScraps();
                            // notify that the database was created and it's ready to be used
                            database.setDatabaseCreated();
                        });
                    }
                })
                .build();

    }



    /**
     * Check whether the database already exists and expose it via {@link #getDatabaseCreated()}
     */
    private void updateDatabaseCreated(final Context context) {
        if (context.getDatabasePath(DATABASE_NAME).exists()) {
            setDatabaseCreated();
        }
    }

    private void setDatabaseCreated(){
        mIsDatabaseCreated.postValue(true);
    }

//    private static void insertData(Context appContext, final ScraperDatabase database, final List<ScrapEntity> scrapEntities) {
//        database.runInTransaction(() -> {
////            getJsonMallScraps(appContext, database);
////            insertOrderRules(appContext, database);
//            database.scrapDao().inserts(scrapEntities);
//
//        });
//    }

    private static void getJsonMallScraps(Context context, ScraperDatabase database) {

        String jsonStr = Utils.loadJSONFromAsset(context, "mall_info.json");

        MallScrapInfo mallScrapInfo;

        try {


            Log.i("array","jsonStr : "+jsonStr);
            mallScrapInfo = Utils.fromJson(jsonStr, MallScrapInfo.class);
            Log.i("getJsonMallScraps", "mall_info : "+mallScrapInfo);

            database.mallDao().insertList(mallScrapInfo.getMalls());
            if (mallScrapInfo.getLoginJsonRules() != null)database.loginApiRuleDao().insert(mallScrapInfo.getLoginJsonRules());
            database.loginWebRuleDao().insertList(mallScrapInfo.getLoginHtmlRules());
            database.orderApiRuleDao().insert(mallScrapInfo.getOrderJsonRules());
            database.orderWebRuleDao().insertList(mallScrapInfo.getOrderHtmlRules());
            Log.i("getJsonMallScraps", "getMalls : "+mallScrapInfo.getMalls());
            Log.i("getJsonMallScraps", "getLoginJsonRules : "+mallScrapInfo.getLoginJsonRules());
            Log.i("getJsonMallScraps", "getLoginHtmlRules : "+mallScrapInfo.getLoginHtmlRules());
            Log.i("getJsonMallScraps", "getOrderJsonRules : "+mallScrapInfo.getOrderJsonRules());
            Log.i("getJsonMallScraps", "getOrderHtmlRules : "+mallScrapInfo.getOrderHtmlRules());



        } catch (Exception e) {
            e.printStackTrace();
        }
    }

//    private static void insertOrderRules(Context mContext, ScraperDatabase database) {
//
//        List<RuleEntity> ruleEntityList = new ArrayList<>();
//        AssetManager am = mContext.getAssets();
//        InputStream inStream;
//        int i  = 0;
//        try {
////            inStream = am.open("order_rules_view.tsv");
//            inStream = am.open("order_rules_by_hyewon.tsv");
//            BufferedReader buffer = new BufferedReader(new InputStreamReader(inStream));
//            String line ;
//            while ((line = buffer.readLine()) != null) {
//                if(i == 0){
//                    i++;
//                    continue;
//                }
//
//                String[] colums = line.split("\t");
//                RuleEntity rule = new RuleEntity(
//                        /*Integer.parseInt(colums[0]),
//                        colums[1],
//                        colums[2],
//                        colums[3],
//                        colums[4],
//                        colums[5],
//                        colums[6],
//                        colums[7],
//                        colums[8],
//                        colums[9],
//                        colums[10],
//                        colums[11],
//                        colums[12],
//                        colums[13]*/
//                        Integer.parseInt(colums[0]),
//                        colums[1],
//                        colums[4],
//                        colums[5],
//                        colums[6],
//                        colums[7],
//                        colums[8],
//                        colums[9],
//                        colums[10],
//                        colums[11],
//                        colums[12],
//                        colums[13],
//                        colums[14],
//                        colums[15]
//                );
//                Log.i("insert_rules","rule : "+rule);
//                ruleEntityList.add(rule);
//
//                i++;
//            }
//
//            if (ruleEntityList.size() > 0) {
//                List<Long> ids = database.ruleDao().insertList(ruleEntityList);
//                Log.i("insertList","ids : "+ids.size());
//            }
//
//
//        } catch (Exception e) {
//            e.printStackTrace();
//        }
//    }

    private static void addDelay() {
        try {
            Thread.sleep(4000);
        } catch (InterruptedException ignored) {
        }
    }

    public LiveData<Boolean> getDatabaseCreated() {
        return mIsDatabaseCreated;
    }

}
