package com.tenqube.visual_scraper;

import android.Manifest;
import android.content.Context;
import android.support.annotation.NonNull;

import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.repository.ScrapRepository;
import com.tenqube.visual_scraper.scrap.OnPurchaseResult;
import com.tenqube.visual_scraper.scrap.Scraper;
import com.tenqube.visual_scraper.utils.CSVHelper;
import com.tenqube.visual_scraper.utils.Utils;
import com.tenqube.visual_scraper.vo.Mall;
import com.tenqube.visual_scraper.vo.ParsingRuleData;
import com.tenqube.visual_scraper.vo.api.RequestDocumentData;

import java.util.List;

import static com.tenqube.visual_scraper.constants.Constants.ERROR.LOGIN_FAIL;
import static com.tenqube.visual_scraper.constants.Constants.ERROR.NOT_FOUND_MALL;

public class ScrapServiceImpl implements ScrapService {

    private final Context context;
    private final String apiKey;
    private final String layer;
    ScrapRepository repository;
    Scraper scraper;


    private boolean isDoing;

    public ScrapServiceImpl(@NonNull Context context, @NonNull String apiKey, @NonNull String layer) {
        this.context = context;
        this.apiKey = apiKey;
        this.layer = layer;
        repository = Injection.provideScrapRepository(context.getApplicationContext());
        repository.saveKey(apiKey);
        repository.saveLayer(layer);
    }

    /**
     * 1. 스크랩 정보 동기화
     * 2. 스크랩 정보 로드
     * 3. 파싱 시작
     * 4.
     * @param userId 사용자 아이디
     * @param userPw 사용자 패스워드
     * @param mall 몰정보
     * @param onResult
     */
    @Override
    public void start(String userId, String userPw, Constants.MALL mall, OnResult onResult) {
        if(isDoing)
            return;

        repository.syncRule(items ->
                repository.getUser(userId, userPw, mall, user -> {

                    if(user == null) {
                        onResult.onFail(mall, LOGIN_FAIL, "사용자 없음 ");
                        return;
                    }
                    start(mall, user, onResult);
        }));
    }

    @Override
    public void start(Constants.MALL mall, OnResult onResult) {
        if(isDoing)
            return;

        repository.syncRule(items ->
            repository.getUser(mall, user -> {

                if(user == null) {
                    onResult.onFail(mall, LOGIN_FAIL, "사용자 없음 ");
                    return;
                }
                start(mall, user, onResult);
        }));
    }

    @Override
    public void signOut(Constants.MALL mall) {
        repository.deleteUser(mall);
    }

    @Override
    public void getMalls(Callback<List<Mall>> callback) {
        repository.getMalls(items -> {
            callback.onDataLoaded(items);
        });
    }

    private void start(@NonNull final Constants.MALL mall, @NonNull final UserEntity userEntity, @NonNull final OnResult onResult) {

        isDoing = true;
        repository.geScrapRule(mall, scrap -> {

            if(scrap == null) {
                onResult.onFail(mall, NOT_FOUND_MALL, "몰정보 없음");
                return;
            }
            ParsingRuleData parsingRuleData = new ParsingRuleData(scrap, userEntity);

            scraper = new Scraper(context, parsingRuleData, new OnPurchaseResult() {

                @Override
                public void onLoginFail(int statusCode, String errorMessage) {
                    isDoing = false;
                    repository.deleteUser(userEntity);
                    onResult.onFail(mall, Constants.ERROR.LOGIN_FAIL, errorMessage);

                }

                @Override
                public void onPurchaseFinished(String statusMessage, RequestDocumentData purchaseHtml) {
                    isDoing = false;
                    repository.requestMallParsing(userEntity.getId(), purchaseHtml, items -> onResult.onSuccess(mall));
                }

                @Override
                public void onFail(int statusCode, String errorMessage) {
                    isDoing = false;
                    onResult.onFail(mall, Constants.ERROR.PARSING, errorMessage);

                }
            }, true);

            scraper.start();
        });

    }

    @Override
    public void getAllProducts(Callback<List<OrderEntity>> callback) {
        repository.getAllOrders(callback);
    }

    @Override
    public void getProducts(Callback<List<OrderEntity>> callback, Constants.MALL... malls) {

    }

    @Override
    public void exportCSV() throws SecurityException {
        String[] perms = {Manifest.permission.WRITE_EXTERNAL_STORAGE};

        if (Utils.hasPermissions(context, perms)) {

            repository.getAllOrders(CSVHelper::export);

        } else {
            throw new SecurityException();
        }

    }
}
