package com.tencent.wecast;


import android.content.Context;
import android.content.SharedPreferences;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.bumptech.glide.load.resource.gif.GifDrawable;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.tencent.wecast.utils.DeviceInfoUtil;
import com.tencent.wecast.utils.GsonUtil;
import com.tencent.wecast.utils.ImageUtil;
import com.tencent.wecast.utils.Logger;
import com.tencent.wecast.utils.StorageUtil;

import java.io.File;
import java.io.FileInputStream;
import java.util.regex.Pattern;

// 投屏UI可配置管理类
public class WeCastUIConfigManager {
    private static final String TAG = "WeCastUIConfigManager";

    public static int FORCE_HIDE_TAG_KEY = 0;

    private static final int CURRENT_UI_VERSION = 1;    // 当前UI版本为1.0

    public static final String PICASSO_PREFIX_TAG = "picasso_";

    private int mStatusBarColor = 0;
    private int mDisplayWidth = 0;
    private int mDisplayHeight = 0;

    private JsonObject mDefaultExtraJson;   // 首页可运营缓存信息


    // 模版ID，标识有多少种activity的layout
    public enum UITemplateId {
        TEMPLATE_1, // 当前仅有一套模版
        TEMPLATE_2,
        TEMPLATE_3,
        TEMPLATE_4,
        TEMPLATE_5
    }

    // activity的标识
    public enum ActivityLabel {
        RECEIVER_SPLASH,
        RECEIVER_WAITING,
        RECEIVER_CASTING,

        SENDER_SPLASH_1,
        SENDER_SPLASH_2,
        SENDER_MAIN_1,
        SENDER_MAIN_2,
        SENDER_GUIDE,
        SENDER_USER_LIST,
        SENDER_USER_ADD
    }

    private static final WeCastUIConfigManager ourInstance = new WeCastUIConfigManager();

    public static WeCastUIConfigManager getInstance() {
        return ourInstance;
    }

    private WeCastUIConfigManager() {
        mDefaultExtraJson = new JsonObject();

        JsonObject pinTitleJson = new JsonObject();
        pinTitleJson.addProperty("color", "0xffe6e6e6");
        pinTitleJson.addProperty("scale", 1);
        pinTitleJson.addProperty("x", "center");
        pinTitleJson.addProperty("y", "0.28");
        mDefaultExtraJson.add("pin_title", pinTitleJson);

        JsonObject pinJson = new JsonObject();
        pinJson.addProperty("color", "0xffffffff");
        pinJson.addProperty("scale", 1);
        pinJson.addProperty("x", "center");
        pinJson.addProperty("y", "center");
        mDefaultExtraJson.add("pin", pinJson);

        JsonObject corpCodeJson = new JsonObject();
        corpCodeJson.addProperty("color", "0xffffffff");
        corpCodeJson.addProperty("scale", 1);
        corpCodeJson.addProperty("x", "0.735");
        corpCodeJson.addProperty("y", "0.751");
        corpCodeJson.addProperty("is_show", true);
        mDefaultExtraJson.add("corp_code", corpCodeJson);


        JsonObject qrJson = new JsonObject();
        qrJson.addProperty("scale", 1);
        qrJson.addProperty("x", "0.35");
        qrJson.addProperty("y", "0.70");
        qrJson.addProperty("is_show", true);
        mDefaultExtraJson.add("qr", qrJson);
    }

    private IConfigUI mUIConfiguration = null;
    private UITemplateId mUITemplateId = UITemplateId.TEMPLATE_1;
    private JsonArray mUIConfigArray = null;
    private JsonObject mOperationInfo = null;   // 首页可运营缓存信息
    private float mPinLastScale = 1.0f;
    private float mPinTitleLastScale = 1.0f;
    private float mCorpCodeLastScale = 1.0f;
    private float mCorpBindQrCodeLastScale = 1.0f;

    private String mRenderBgUrl = "";

    // 获取UI版本号
    public int getUIVersion() {
        return CURRENT_UI_VERSION;
    }

    // 初始化UI ID映射
    public void initialUIIdMapping(IConfigUI configuration) {
        mUIConfiguration = configuration;

        DisplayMetrics metrics = DeviceInfoUtil.getDisplayMetrics(WeCastAppContext.get());
        if (metrics != null) {
            mDisplayWidth = metrics.widthPixels;
            mDisplayHeight = metrics.heightPixels;
        }
    }

    // 发送端编译构建时配置TemplateID(静态）
    public void setTemplateId(UITemplateId id) {
        mUITemplateId = id;
    }


    // 从本地更新config配置（sp中缓存的配置）
    public void readLocalConfig() {
        JsonObject configJson = getTestConfigJson();
        if (configJson != null) {
            // 本地如果存在彩蛋的配置文件优先采用该配置
            initConfig(configJson);
        } else {
            String initConfig = getConfigFromSp();
            Logger.t(TAG).d("load config from sp: " + initConfig);
            if (!TextUtils.isEmpty(initConfig)) {
                try {
                    JsonObject lastConfig = GsonUtil.getGson().fromJson(initConfig, JsonObject.class);
                    initConfig(lastConfig);
                } catch (Throwable e) {
                    Logger.t(TAG).e("failure: " + e.getMessage());
                }
            }
            initOperation();
        }
    }

    public int getStatusBarColor(int srcColor) {
        if (mStatusBarColor != 0) {
            return mStatusBarColor;
        }
        return srcColor;
    }

    public int getMainBgResId() {
        if (mUIConfiguration != null) {
            return mUIConfiguration.getMainBgResId(mUITemplateId);
        }
        return 0;
    }

    public String getRenderBgUrl() {
        return mRenderBgUrl;
    }

    public View getLayoutRootView(WeCastUIConfigManager.ActivityLabel label,
                                  LayoutInflater inflater, ViewGroup container) {
        try {
            return inflater.inflate(getLayoutId(label), container, false);
        } catch (Exception e) {
            Logger.t(TAG).e("LayoutInflater Exception:" + e.toString());
            e.printStackTrace();
            return null;
        }
    }

    public View getLayoutRootView(WeCastUIConfigManager.ActivityLabel label, Context context) {
        try {
            LayoutInflater inflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
            return inflater.inflate(getLayoutId(label), null);
        } catch (Exception e) {
            Logger.t(TAG).e("LayoutInflater Exception:" + e.toString());
            e.printStackTrace();
            return null;
        }
    }

    public void applyOperationView(ActivityLabel activityLabel,
                                   ImageView ivBg, TextView tvContent,
                                   TextView tvPinTitle, TextView tvPin,
                                   TextView tvCorpCode, ImageView ivCorpBindQrCode) {
        try {
            if (mOperationInfo != null) {
                String url = mOperationInfo.get("url").getAsString().trim();
                String text = mOperationInfo.get("text").getAsString().trim();
                String extra = mOperationInfo.get("extra").getAsString();

                if (!TextUtils.isEmpty(url)) {
                    JsonObject urlObj = new JsonObject();
                    urlObj.addProperty("type", "image");
                    urlObj.addProperty("src", url);
                    apply2Item(activityLabel, urlObj, "", true, ivBg);
                }

                if (!TextUtils.isEmpty(text)) {
                    JsonObject textObj = new JsonObject();
                    textObj.addProperty("type", "text");
                    textObj.addProperty("text", text);
                    textObj.addProperty("is_show", true);
                    apply2Item(activityLabel, textObj, "", true, tvContent);
                } else {
                    JsonObject textObj = new JsonObject();
                    textObj.addProperty("type", "text");
                    textObj.addProperty("text", text);
                    textObj.addProperty("is_show", false);
                    apply2Item(activityLabel, textObj, "", false, tvContent);
                }

                if (!TextUtils.isEmpty(extra)) {
                    JsonObject extraObj = GsonUtil.getGson()
                            .fromJson(extra, JsonObject.class);
                    if (extraObj.has("pin")) {
                        copyOpsUIConfig(activityLabel, tvPin, extraObj, "pin");
                    }

                    if (extraObj.has("pin_title")) {
                        copyOpsUIConfig(activityLabel, tvPinTitle, extraObj, "pin_title");
                    }

                    if (extraObj.has("corp_code")) {
                        copyOpsUIConfig(activityLabel, tvCorpCode, extraObj, "corp_code");
                    }

                    if (extraObj.has("qr")) {
                        copyOpsUIConfig(activityLabel, ivCorpBindQrCode, extraObj, "qr");
                    }
                } else {
                    copyOpsUIConfig(activityLabel, tvPin, mDefaultExtraJson, "pin");
                    copyOpsUIConfig(activityLabel, tvPinTitle, mDefaultExtraJson, "pin_title");
                    copyOpsUIConfig(activityLabel, tvCorpCode, mDefaultExtraJson, "corp_code");
                    copyOpsUIConfig(activityLabel, ivCorpBindQrCode, mDefaultExtraJson, "qr");
                }
            }
        } catch (Exception e) {
            Logger.t(TAG).e("applyOperationView e = " + e);
        }
    }

    public void applyDefaultOperationView(ActivityLabel activityLabel,
                                          TextView tvPinTitle, TextView tvPin,
                                          TextView tvCorpCode, ImageView ivCorpBindQrCode) {
        try {
            copyOpsUIConfig(activityLabel, tvPin, mDefaultExtraJson, "pin");
            copyOpsUIConfig(activityLabel, tvPinTitle, mDefaultExtraJson, "pin_title");
            copyOpsUIConfig(activityLabel, tvCorpCode, mDefaultExtraJson, "corp_code");
            copyOpsUIConfig(activityLabel, ivCorpBindQrCode, mDefaultExtraJson, "qr");
        } catch (Exception e) {
            Logger.t(TAG).e("applyDefaultOperationView e = " + e);
        }
    }

    // 将首页可运营返回值转为UI可配置格式
    private void copyOpsUIConfig(ActivityLabel activityLabel, View view,
                                 JsonObject extraObj, String key) {
        JsonObject origin;
        if (extraObj == null) {
            // 首页可运营元素重置为默认的位置和大小
            origin = new JsonObject();
            origin.addProperty("x", "center");
            if ("pin_title".equals(key)) {
                origin.addProperty("y", "0.21");
            } else if ("pin".equals(key)) {
                origin.addProperty("y", "center");
            }
            origin.addProperty("scale", "1.0");
        } else {
            origin = extraObj.getAsJsonObject(key);
        }

        JsonObject dst = new JsonObject();
        if ("pin".equals(key)) {
            dst.addProperty("type", "text");
        } else if ("pin_title".equals(key)) {
            dst.addProperty("type", "text");
        } else if ("corp_code".equals(key)) {
            dst.addProperty("type", "text");
        } else if ("qr".equals(key)) {
            dst.addProperty("type", "image");
        } else {
            dst.addProperty("type", "text");
        }

        if (origin.has("color")) {
            dst.addProperty("text_color", origin.get("color").getAsString());
        }

        if (origin.has("text")) {
            dst.addProperty("text", origin.get("text").getAsString());
        }

        if (origin.has("x") && origin.has("y")) {
            dst.addProperty("pos",
                    origin.get("x").getAsString()
                            + ","
                            + origin.get("y").getAsString());
        }
        if (origin.has("scale")) {
            float scale = origin.get("scale").getAsFloat();
            if ("pin".equals(key)) {
                float tmp = mPinLastScale;
                mPinLastScale = scale;
                scale = scale / tmp;
            } else if ("pin_title".equals(key)) {
                float tmp = mPinTitleLastScale;
                mPinTitleLastScale = scale;
                scale = scale / tmp;
            } else if ("corp_code".equals(key)) {
                float tmp = mCorpCodeLastScale;
                mCorpCodeLastScale = scale;
                scale = scale / tmp;
            } else if ("qr".equals(key)) {
                float tmp = mCorpBindQrCodeLastScale;
                mCorpBindQrCodeLastScale = scale;
                scale = scale / tmp;
            }
            dst.addProperty("scale", String.valueOf(scale));
        }

        boolean isShow = true;
        if (origin.has("is_show")) {
            isShow = origin.get("is_show").getAsBoolean();
        }

        dst.addProperty("is_show", isShow);
        apply2Item(activityLabel, dst, "", isShow, view);
    }

    public void saveOperationInfo(String url, String text, String extra) {
        JsonObject data = new JsonObject();
        data.addProperty("url", url);
        data.addProperty("text", text);
        data.addProperty("extra", extra);
        String opsInfo = GsonUtil.getGson().toJson(data);
        mOperationInfo = GsonUtil.getGson().fromJson(opsInfo, JsonObject.class);

        SharedPreferences sp =
                WeCastAppContext.get().getSharedPreferences(getConfigSpName(), Context.MODE_PRIVATE);
        SharedPreferences.Editor edit = sp.edit();
        edit.putString("ops_config", opsInfo);
        edit.apply();
    }

    private void initOperation() {
        SharedPreferences sp =
                WeCastAppContext.get().getSharedPreferences(getConfigSpName(), Context.MODE_PRIVATE);
        mOperationInfo = GsonUtil.getGson()
                .fromJson(sp.getString("ops_config", ""), JsonObject.class);
    }

    private int getLayoutId(ActivityLabel label) {
        if (mUIConfiguration != null) {
            return mUIConfiguration.getLayoutId(mUITemplateId, label);
        }
        return 0;
    }

    // 通过native层上抛的信息来更新配置信息
    // "{\"template_id\":1}" native传上来的是缺少{'ui':{'config':"{native传出的内容部分}"}}
    public void updateConfig(String configTxt) {
        if (!TextUtils.isEmpty(configTxt)) {
            JsonObject data = new JsonObject();
            JsonObject ui = new JsonObject();
            ui.addProperty("config", configTxt);
            data.add("ui", ui);
            JsonObject config = GsonUtil.getGson().fromJson(configTxt, JsonObject.class);
            if (config != null
                    && config.has("template_id")) {
                saveUIConfigToSp(GsonUtil.getGson().toJson(data));
                initConfig(data);
                return;
            }
        }
        saveUIConfigToSp("");
        initConfig(null);
    }

    public void applyUIConfig(ActivityLabel activityLabel, View v) {
        if (mUIConfigArray == null || mUIConfiguration == null || v == null) {
            return;
        }
        Logger.t(TAG).d("apply ui config > " + activityLabel);
        applyUIConfigImpl(activityLabel, v, mUIConfigArray);
    }

    // FIXME(mowang): 先为tencent做定制逻辑，2.1.0版本做通用的设计
    public void applyTencentGuide(View view, boolean isMOA) {
        if (mUIConfigArray == null || mUIConfiguration == null || view == null) {
            return;
        }

        ActivityLabel label = WeCastUIConfigManager.ActivityLabel.RECEIVER_CASTING;
        try {
            for (JsonElement element : mUIConfigArray) {
                JsonObject item = element.getAsJsonObject();
                String id = item.get("id").getAsString();
                if (id.equals("array_rk_guide")) {
                    JsonObject src = item.getAsJsonObject("src");
                    if (src.has("moa")) {
                        JsonArray arrSrc = null;
                        if (isMOA) {
                            arrSrc = src.getAsJsonArray("moa");
                        } else {
                            arrSrc = src.getAsJsonArray("wework");
                        }
                        applyUIConfigImpl(label, view, arrSrc);
                    }
                    break;
                }
            }
        } catch (Exception e) {
            Logger.t(TAG).e("Exception:" + e.toString());
        }
    }

    private void applyUIConfigImpl(ActivityLabel activityLabel, View v, JsonArray jsonArray) {
        try {
            for (JsonElement element : jsonArray) {
                JsonObject item = element.getAsJsonObject();
                String id = item.get("id").getAsString();
                boolean isShowSet = false;
                boolean show = true;
                if (item.has("is_show")) {
                    isShowSet = true;
                    show = item.get("is_show").getAsBoolean();
                }

                if (item.has("is_group") && item.get("is_group").getAsBoolean()) {
                    int[] viewIds = mUIConfiguration.getViewIds(mUITemplateId, activityLabel, id);
                    if (viewIds != null) {
                        for (int resId : viewIds) {
                            View view = v.findViewById(resId);
                            if (!isShowSet) {
                                show = view.getVisibility() == View.VISIBLE;
                            }
                            try {
                                apply2Item(activityLabel, item, id, show, view);
                            } catch (Exception ignore) {
                            }
                        }
                    }
                } else {
                    int resId = mUIConfiguration.getViewId(mUITemplateId, activityLabel, id);
                    if (resId == 0) {
                        continue;
                    }
                    View view = v.findViewById(resId);
                    if (!isShowSet) {
                        show = view.getVisibility() == View.VISIBLE;
                    }
                    try {
                        apply2Item(activityLabel, item, id, show, view);
                    } catch (Exception ignore) {
                    }
                }
            }
        } catch (Exception e) {
            Logger.t(TAG).e("Exception:" + e.toString());
        }
    }

    private void apply2Item(ActivityLabel activityLabel, JsonObject item, String id, boolean show, final View view)
            throws NullPointerException, IllegalArgumentException {

        Boolean forceHide = false;
        try {
            if(FORCE_HIDE_TAG_KEY != 0) {
                Object tag = view.getTag(FORCE_HIDE_TAG_KEY);
                if (tag != null) {
                    forceHide = (Boolean) tag;
                }
            }
        } catch (Exception e) {
            forceHide = false;
        }


        if (show && !forceHide) {
            view.setVisibility(View.VISIBLE);
        } else {
            view.setVisibility(View.INVISIBLE);
        }
        String type = "";
        if (item.has("type")) {
            type = item.get("type").getAsString();
        }

        if ("text".equals(type)) {
            TextView txtView = (TextView) view;
            if (item.has("scale")) {
                float textSize = txtView.getTextSize();
                float scale = item.get("scale").getAsFloat();
                txtView.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSize * scale);
            }
            if (item.has("text_color")) {
                txtView.setTextColor(parseColor(item.get("text_color").getAsString()));
            }
            if (item.has("text")) {
                txtView.setText(item.get("text").getAsString());
            }
            if (item.has("text_all")) {
                txtView.setText(getLocaleText(item.get("text_all").getAsJsonObject()));
            }
            if (item.has("src")) {
                String srcValue = item.get("src").getAsString();
                if (srcValue.contains("0x")) {
                    view.setBackgroundColor(parseColor(srcValue));
                }
            }
        } else if ("button".equals(type)) {
            Button btn = (Button) view;
            if (item.has("src")) {
                String srcValue = item.get("src").getAsString();
                if (srcValue.contains("0x")) { // 若传递的是颜色值
                    btn.setBackgroundColor(parseColor(srcValue));
                } else {
                    setBgRes(btn, mUIConfiguration.getResourceId(mUITemplateId, srcValue));
                }
            }
            if (item.has("text")) {
                btn.setText(item.get("text").getAsString());
            }
            if (item.has("text_all")) {
                btn.setText(getLocaleText(item.get("text_all").getAsJsonObject()));
            }
            if (item.has("style")) {
                int style = mUIConfiguration.getResourceId(mUITemplateId, item.get("style").getAsString());
                if (style > 0) {
                    btn.setTextAppearance(WeCastAppContext.get(), style);
                }
            }
        } else if ("image".equals(type)) {
            if (item.has("src")) {
                String srcValue = item.get("src").getAsString();
                if (srcValue.contains("0x")) { // 若传递的是颜色值
                    view.setBackgroundColor(parseColor(srcValue));
                    // 更改状态栏颜色与背景颜色一致
                    if ("main_background".equalsIgnoreCase(id)) {
                        mStatusBarColor = parseColor(srcValue);
                    }
                } else if (srcValue.startsWith("http")) {
                    if (srcValue.substring(srcValue.lastIndexOf(".") + 1).equals("gif")) {
                        ImageUtil.loadNetWorkImage2view((ImageView) view, srcValue, true
                                , new ImageUtil.OnNetWorkImageLoadedListener() {
                                    @Override
                                    public boolean onSucceed(Drawable resource) {
                                        Logger.t(TAG).d("Glide load gif picture succeed,DrawableClassName:"
                                                + resource.getClass().getCanonicalName());
                                        return false;
                                    }

                                    @Override
                                    public void onFailed(String errorMsg) {
                                        Logger.t(TAG).d("Glide load gif picture failed,errorMessage:" + errorMsg);
                                    }
                                });
                    } else {
                        final Drawable originalDrawable = ((ImageView) view).getDrawable();
                        ImageUtil.loadNetWorkImage2view((ImageView) view, srcValue, false
                                , new ImageUtil.OnNetWorkImageLoadedListener() {
                                    @Override
                                    public boolean onSucceed(Drawable resource) {
                                        Logger.t(TAG).d("Glide load picture succeed,errorMessage:"
                                                + resource.getClass().getCanonicalName());
                                        if (resource instanceof GifDrawable) {
                                            ((ImageView) view).setImageDrawable(originalDrawable);
                                            return true;
                                        }
                                        return false;
                                    }

                                    @Override
                                    public void onFailed(String errorMsg) {
                                        Logger.t(TAG).d("Glide load picture failed,errorMessage:" + errorMsg);
                                    }
                                });
                    }
                } else {
                    int resourceId = mUIConfiguration.getResourceId(mUITemplateId, srcValue);
                    if (resourceId != 0) {
                        ((ImageView) view).setImageResource(resourceId);
                    }
                }
            }

            if (item.has("scale")) {
                float scale = item.get("scale").getAsFloat();
                ViewGroup.LayoutParams lp = view.getLayoutParams();
                lp.width = (int) (lp.width * scale);
                lp.height = (int) (lp.height * scale);
                view.setLayoutParams(lp);
            }
        } else if ("editor".equals(type)) {
            EditText et = (EditText) view;
            if (item.has("text_color")) {
                et.setTextColor(parseColor(item.get("text_color").getAsString()));
            }
            if (item.has("view_color")) {
                et.setBackgroundColor(parseColor(item.get("view_color").getAsString()));
            }
            if (item.has("hint_color")) {
                et.setHintTextColor(parseColor(item.get("hint_color").getAsString()));
            }
        } else if ("view".equals(type)) { // view
            if (item.has("src")) {
                setBgRes(view, mUIConfiguration.getResourceId(mUITemplateId, item.get("src").getAsString()));
            }
            if (item.has("view_color")) {
                view.setBackgroundColor(parseColor(item.get("view_color").getAsString()));
            }
        } else if ("list_view".equals(type)) { // list_view
            if (item.has("divider_color")) {
                ListView lv = (ListView) view;
                lv.setDivider(new ColorDrawable(parseColor(item.get("divider_color").getAsString())));
                // ListView设置divider后要重新设置高度 不然高度会被置回-1
                lv.setDividerHeight(1);
            }
        } else { // view_group
            if (item.has("child_view")) {
                JsonArray childViewArray = item.get("child_view").getAsJsonArray();
                applyUIConfigImpl(activityLabel, view, childViewArray);
            }
        }

        boolean hasPos = item.has("pos");
        if (hasPos) {
            ViewGroup.LayoutParams lp = view.getLayoutParams();
            if (lp instanceof RelativeLayout.LayoutParams) {
                RelativeLayout.LayoutParams newLp =
                        new RelativeLayout.LayoutParams(lp.width, lp.height);
                if (item.has("pos")) {
                    String[] xy = item.get("pos").getAsString().split(",");
                    if (xy.length == 2) {
                        if ("center".equals(xy[0])) {
                            newLp.addRule(RelativeLayout.CENTER_HORIZONTAL);
                        } else if (mDisplayWidth != 0 && isDoubleOrFloat(xy[0])) {
                            float fw = Float.valueOf(xy[0]);
                            if (0 < fw && fw <= 1) {
                                newLp.leftMargin = (int) (fw * mDisplayWidth);
                            }
                        } else if (isNumeric(xy[0])) {
                            newLp.leftMargin = Integer.valueOf(xy[0]);
                        }

                        if ("center".equals(xy[1])) {
                            newLp.addRule(RelativeLayout.CENTER_VERTICAL);
                        } else if (mDisplayHeight != 0 && isDoubleOrFloat(xy[1])) {
                            float fw = Float.valueOf(xy[1]);
                            if (0 < fw && fw <= 1) {
                                newLp.topMargin = (int) (fw * mDisplayHeight);
                            }
                        } else if (isNumeric(xy[1])) {
                            newLp.topMargin = Integer.valueOf(xy[1]);
                        }
                    }
                }
                view.setLayoutParams(newLp);
            }
        }
    }

    private String getLocaleText(JsonObject obj) {
        int languageType = WeCastConfigManager.getInstance().getLanguageType();
        String txt = "";
        try {
            switch (languageType) {
                case WeCastConstant.LanguageType.ZH_TW:
                    txt = obj.get("tw").getAsString();
                    break;
                case WeCastConstant.LanguageType.EN_US:
                    txt = obj.get("en").getAsString();
                    break;
                default:
                    txt = obj.get("cn").getAsString();
                    break;
            }
        } catch (Exception ignore) {
        }
        return txt;
    }

    private void setBgRes(View view, int resId) {
        if (resId > 0) {
            view.setBackgroundResource(resId);
        }
    }

    // 读取sdcard/tencent/Wmp/packageName/test_cloudxx.config中配置的ui信息
    private JsonObject getTestConfigJson() {
        Logger.t(TAG).d("local config env: " + WeCastEnv.getCurrentEnv());
        String localFileName = "test_cloud.config";
        switch (WeCastEnv.getCurrentEnv()) {
            case WeCastEnv.ENV_PRE_RELEASE:
                localFileName = "test_cloud_pre.config";
                break;
            case WeCastEnv.ENV_DEVELOP:
                localFileName = "test_cloud_dev.config";
                break;
        }

        String filePath = StorageUtil.getWmpExternalPath(WeCastAppContext.get(), localFileName);
        File testConfigFile = new File(filePath);
        if (testConfigFile.exists()) {
            Logger.t(TAG).d("load local config from file:" + localFileName);
            Long fileLength = testConfigFile.length();
            byte[] contentBytes = new byte[fileLength.intValue()];
            try {
                FileInputStream in = new FileInputStream(testConfigFile);
                in.read(contentBytes);
                in.close();

                String content = new String(contentBytes);
                return GsonUtil.getGson().fromJson(content, JsonObject.class);
            } catch (Exception e) {
                Logger.t(TAG).e("Exception:" + e.toString());
            }
        }
        return null;
    }

    private void initConfig(JsonObject lastConfig) {
        if (lastConfig != null) {
            JsonObject ui = lastConfig.get("ui").getAsJsonObject();
            if (ui.has("config")) {
                ui = GsonUtil.getGson().fromJson(ui.get("config").getAsString(), JsonObject.class);
                mUITemplateId = getUITemplateId(ui.get("template_id").getAsInt());
                mUIConfigArray = ui.get("template_config").getAsJsonArray();
            } else {
                mUITemplateId = getUITemplateId(ui.get("template_id").getAsInt());
                mUIConfigArray = ui.get("template_config").getAsJsonArray();
            }
        } else {
            mUITemplateId = UITemplateId.TEMPLATE_1;
            mUIConfigArray = null;
        }
        initRenderBg();
    }

    private void initRenderBg() {
        if (mUIConfigArray != null) {
            try {
                for (JsonElement element : mUIConfigArray) {
                    JsonObject item = element.getAsJsonObject();
                    String id = item.get("id").getAsString();
                    if ("img_xcast_render_bg".equals(id)) {
                        String src = item.get("src").getAsString();
                        if (src.startsWith("http")) {
                            mRenderBgUrl = src;
                            break;
                        }
                    }
                }
            } catch (Exception e) {
                Logger.t(TAG).e("init render background url error: " + e.getMessage());
            }
        }
    }

    private String getConfigFromSp() {
        SharedPreferences sp =
                WeCastAppContext.get().getSharedPreferences(getConfigSpName(), Context.MODE_PRIVATE);
        return sp.getString("ui_config", "");
    }

    private void saveUIConfigToSp(String uiConfig) {
        SharedPreferences sp =
                WeCastAppContext.get().getSharedPreferences(getConfigSpName(), Context.MODE_PRIVATE);
        SharedPreferences.Editor edit = sp.edit();
        edit.putString("ui_config", uiConfig);
        edit.apply();
    }

    private String getConfigSpName() {
        return "wecast_config" + WeCastEnv.getCurrentEnv();
    }

    private UITemplateId getUITemplateId(int id) {
        switch (id) {
            case 1:
                return UITemplateId.TEMPLATE_1;
            case 2:
                return UITemplateId.TEMPLATE_2;
            case 3:
                return UITemplateId.TEMPLATE_3;
            case 4:
                return UITemplateId.TEMPLATE_4;
            case 5:
                return UITemplateId.TEMPLATE_5;
            default:
                return UITemplateId.TEMPLATE_1;
        }
    }

    private int parseColor(String hexVal) {
        try {
            String colorHex = hexVal.replace("0x", "#");
            return Color.parseColor(colorHex);
        } catch (Exception e) {
            Logger.t(TAG).e("parseColor hexVal = " + hexVal + " | e = " + e);
            return Color.BLACK;
        }
    }

    public interface IConfigUI {
        int getLayoutId(UITemplateId templateId, ActivityLabel label);

        int getViewId(UITemplateId templateId, ActivityLabel label, String viewLabel);

        int getResourceId(UITemplateId templateId, String resLabel);

        int getMainBgResId(UITemplateId templateId);

        int[] getViewIds(UITemplateId templateId, ActivityLabel label, String viewLabel);
    }

    /*
     * 是否为浮点数？double或float类型。
     * @param str 传入的字符串。
     * @return 是浮点数返回true,否则返回false。
     */
    private static boolean isDoubleOrFloat(String str) {
        Pattern pattern = Pattern.compile("^[-+]?[.\\d]*$");
        return pattern.matcher(str).matches();
    }

    private static boolean isNumeric(String str) {
        Pattern pattern = Pattern.compile("[0-9]*");
        return pattern.matcher(str).matches();
    }

}
