package com.tencent.wecast.jni;

import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.widget.Toast;

import com.tencent.wecast.WeCastAppContext;
import com.tencent.wecast.WeCastDeviceManager;
import com.tencent.wecast.utils.Logger;
import com.tencent.wecast.utils.StorageUtil;
import com.tencent.wecast.utils.WeCastUtil;

import java.io.IOException;

/**
 * java提供给c++的通用函数服务
 */
public abstract class BaseJ2CppService {
    private final static String TAG = "BaseJ2CppService";

    // 获取目录
    private static final int TYPE_GET_DIR_LOG = 0;
    private static final int TYPE_GET_DIR_TMP = 1;
    private static final int TYPE_GET_DIR_CONFIG = 2;
    private static final int TYPE_GET_DIR_DB = 3;
    // 获取设备信息
    private static final int TYPE_DEVICE_INFO_MODEL = 0;
    protected static final int TYPE_DEVICE_INFO_ROM_VERSION = 1;
    private static final int TYPE_DEVICE_INFO_GPU_DRIVER_VERSION = 2;
    private static final int TYPE_DEVICE_INFO_GPU_INFO = 3;
    private static final int TYPE_DEVICE_INFO_CPU_INFO = 4;
    private static final int TYPE_DEVICE_INFO_NETWORK_TYPE = 5;
    private static final int TYPE_DEVICE_INFO_WIFI_MAC = 6;
    private static final int TYPE_DEVICE_INFO_AP_SSID = 7;
    private static final int TYPE_DEVICE_INFO_AP_MAC = 8;
    private static final int TYPE_DEVICE_INFO_WIRED_MAC = 9;
    private static final int TYPE_DEVICE_INFO_RAM_INFO = 10;

    private Handler mMainHandler;

    protected BaseJ2CppService() {
    }

    public void register() {
        mMainHandler = new Handler(Looper.getMainLooper());
        registerImpl();
    }

    /**
     * 发送端和接收端对于c++上抛的事件不同处理逻辑
     */
    protected abstract void registerEBHandle();

    /**
     * 实际注册提供给c++调用的函数
     * 子类需要调用super.registerImpl(),再根据实际不同需求给c++提供能力
     */
    protected void registerImpl() {
        registerEBHandle();

        JniHelper.registerFunc4Cpp("GetXcastConfig", new JniHelper.OnCppCall() {
            @Override
            public ParamsPackage.ParamsToCpp handle(ParamsPackage.ParamsFromCpp params) {
                try {
                    ParamsPackage.ParamsToCpp retParam = new ParamsPackage.ParamsToCpp();
                    retParam.addString(StorageUtil.getXCastConfFilePath(JniHelper.getAppContext()));
                    return retParam;
                } catch (IOException e) {
                    e.printStackTrace();
                }
                return null;
            }
        });

        // 提供cpp把任务执行在UI线程
        JniHelper.registerFunc4Cpp("runOnUIThread",
                new JniHelper.OnCppCall() {
                    @Override
                    public ParamsPackage.ParamsToCpp handle(ParamsPackage.ParamsFromCpp params) {
                        try {
                            long looperAddress = (long) params.getNext();
                            //Logger.t(TAG).d("runOnUIThread Looper address: " + looperAddress);
                            final ParamsPackage.ParamsToCpp looperParams = new ParamsPackage.ParamsToCpp();
                            looperParams.addLong(looperAddress);
                            mMainHandler.post(new Runnable() {
                                @Override
                                public void run() {
                                    JniHelper.callJniFunc("runMainLooper", looperParams);
                                }
                            });
                        } catch (IOException e) {
                            Logger.t(TAG).e("runOnUIThread exception: ", e);
                            e.printStackTrace();
                        }
                        return null;
                    }
                });

        // 提供cpp不同类型的目录路径
        // warning: native层不需要尾部的'/'
        JniHelper.registerFunc4Cpp("GetDirByType",
                new JniHelper.OnCppCall() {
                    @Override
                    public ParamsPackage.ParamsToCpp handle(ParamsPackage.ParamsFromCpp params) {
                        try {
                            int type = (int) params.getNext();
                            final ParamsPackage.ParamsToCpp ret = new ParamsPackage.ParamsToCpp();
                            switch (type) {
                                case TYPE_GET_DIR_LOG:
                                    ret.addString(StorageUtil.getWmpLogDirPath(WeCastAppContext.get()));
                                    break;
                                case TYPE_GET_DIR_TMP:
                                    ret.addString(StorageUtil.getWmpTempPath());
                                    break;
                                case TYPE_GET_DIR_CONFIG:
                                    ret.addString(StorageUtil.getWmpConfigDirPath(WeCastAppContext.get()));
                                case TYPE_GET_DIR_DB:
                                    ret.addString(StorageUtil.getWmpDbDirPath(WeCastAppContext.get()));
                                default:
                                    ret.addString(StorageUtil.getWmpDirPath(WeCastAppContext.get(), "wmp"));
                                    break;
                            }
                            return ret;
                        } catch (IOException e) {
                            Logger.t(TAG).e("[dir] GetDirByType exception: ", e);
                            e.printStackTrace();
                        }
                        return null;
                    }
                });
        JniHelper.registerFunc4Cpp("setCaptureProperty", new JniHelper.OnCppCall() {
            @Override
            public ParamsPackage.ParamsToCpp handle(ParamsPackage.ParamsFromCpp params) {
                WeCastUtil.setCaptureData();
                return null;
            }
        });
        JniHelper.registerFunc4Cpp("isPackageNameExist",
                new JniHelper.OnCppCall() {
                    @Override
                    public ParamsPackage.ParamsToCpp handle(ParamsPackage.ParamsFromCpp params) {
                        try {
                            String pn = (String) params.getNext();
                            if (TextUtils.isEmpty(pn)) {
                                Logger.t(TAG).d("params packageName is null");
                                return null;
                            }
                            PackageManager pm = WeCastAppContext.get().getPackageManager();
                            PackageInfo pInfo = pm.getPackageInfo(pn, PackageManager.GET_ACTIVITIES);

                            final ParamsPackage.ParamsToCpp ret = new ParamsPackage.ParamsToCpp();
                            if (pInfo == null) {
                                Logger.t(TAG).d("packageName:" + pn + " is not exist");
                                ret.addInt(0);
                            } else {
                                ret.addInt(1);
                                Logger.t(TAG).d("packageName:" + pn + " is exist");
                            }
                            return ret;
                        } catch (IOException | PackageManager.NameNotFoundException e) {
                            Logger.t(TAG).e("isPackageNameExist exception: ", e.toString());
                            e.printStackTrace();
                        }
                        return null;
                    }
                });

        JniHelper.registerFunc4Cpp("showToast", new JniHelper.OnCppCallOnUI() {
            @Override
            public ParamsPackage.ParamsToCpp handle(ParamsPackage.ParamsFromCpp params) {
                try {
                    Toast.makeText(JniHelper.getAppContext(),
                            (String) params.getNext(),
                            Toast.LENGTH_LONG).show();
                } catch (IOException e) {
                    e.printStackTrace();
                }
                return null;
            }
        });
    }

    protected ParamsPackage.ParamsToCpp getDeviceInfoParams(int type) {
        try {
            final ParamsPackage.ParamsToCpp ret = new ParamsPackage.ParamsToCpp();
            switch (type) {
                case TYPE_DEVICE_INFO_MODEL:
                    ret.addString(WeCastDeviceManager.getInstance().model);
                    break;
                case TYPE_DEVICE_INFO_ROM_VERSION:
                    ret.addString(WeCastDeviceManager.getInstance().osVersion);
                    break;
                case TYPE_DEVICE_INFO_GPU_DRIVER_VERSION:
                    ret.addString("unknown");
                    break;
                case TYPE_DEVICE_INFO_GPU_INFO:
                    ret.addString(WeCastDeviceManager.getInstance().gpu);
                    break;
                case TYPE_DEVICE_INFO_CPU_INFO:
                    ret.addString(WeCastDeviceManager.getInstance().cpu);
                    break;
                case TYPE_DEVICE_INFO_NETWORK_TYPE:
                    ret.addString(WeCastDeviceManager.getInstance().netType);
                    break;
                case TYPE_DEVICE_INFO_WIFI_MAC:
                    ret.addString(WeCastDeviceManager.getInstance().mac);
                    break;
                case TYPE_DEVICE_INFO_AP_SSID:
                    ret.addString(WeCastDeviceManager.getInstance().wfSSID);
                    break;
                case TYPE_DEVICE_INFO_AP_MAC:
                    ret.addString(WeCastDeviceManager.getInstance().wfMac);
                    break;
                case TYPE_DEVICE_INFO_WIRED_MAC:
                    ret.addString(WeCastDeviceManager.getInstance().wiredMac);
                    break;
                case TYPE_DEVICE_INFO_RAM_INFO:
                    ret.addString(WeCastDeviceManager.getInstance().ram);
                    break;
                default:
                    ret.addString("unknown(" + type + ")");
                    break;
            }
            return ret;
        } catch (IOException e) {
            Logger.t(TAG).e("[device] getDeviceInfo exception: ", e);
            e.printStackTrace();
        }
        return null;
    }
}