package com.tencent.wecast.sender.cloud.activity

import android.app.Dialog
import android.content.pm.PackageManager
import android.os.Build
import android.support.v4.app.ActivityCompat
import android.support.v4.content.ContextCompat
import android.text.TextUtils
import android.widget.Toast
import com.tencent.wecast.WeCastConfigManager
import com.tencent.wecast.WeCastUIConfigManager
import com.tencent.wecast.sender.cloud.R
import com.tencent.wecast.sender.cloud.bean.WeCastInfo
import com.tencent.wecast.sender.cloud.utils.CloudUtil
import com.tencent.wecast.sender.cloud.utils.DownloadApkUtils
import com.tencent.wecast.sender.cloud.widget.DownLoadDialog
import com.tencent.wecast.sender.eb.SenderEvents
import com.tencent.wecast.utils.DeviceInfoUtil
import com.tencent.wecast.utils.EventBusUtil
import com.tencent.wecast.utils.Logger
import org.greenrobot.eventbus.Subscribe
import org.greenrobot.eventbus.ThreadMode
import java.util.*

abstract class CloudLoginActivity : BaseActivity() {

    companion object {

        private const val TAG = "CloudLoginActivity"

        const val DEVICE_NAME_KEY = "device_name"
        const val PERMISSION_REQUEST_CODE = 10001
        const val CAMERA_PERMISSION_REQUEST_CODE = 10002
        const val CORP_ID = "corpId"
        const val CORP_NAME = "corpName"
        const val TOKEN_KEY = "tokenKey"
        const val ALREADY_SET_URL = "alreadySetUrl"
    }

    private var mDelayFinishFlag: Boolean = false
    private var mLoginFinishFlag: Boolean = true

    private var mLoginResult: Boolean = false
    private var mLoginInfo: String? = null
    private var mReturnCode: Int = 0
    protected var mNeedUpdate: Boolean = false

    protected var mFatalDialog: Dialog? = null
    private var mRequestPermissionSuccess: Boolean? = null
    private val mPermissionToRequest = ArrayList<String>()
    private var mIsShowRequestPermissionDialog = false
    private var mPermissionMap: HashMap<String, String>? = null
    private var mOnRequestPermissionListener: OnRequestPermissionListener? = null
    protected var mDownloadDialog: DownLoadDialog? = null

    /**
     * 闪屏最少显示时间
     */
    protected var mSplashShowMinTime = 1000

    interface OnRequestPermissionListener {
        fun onRequestSuccess()
    }

    protected fun requestPermission(permissionsMap: HashMap<String, String>
                                    , onRequestPermissionListener: OnRequestPermissionListener) {
        /** 申请权限  */
        if (Build.VERSION.SDK_INT >= 23) {
            mPermissionToRequest.clear()
            mPermissionMap = permissionsMap
            mOnRequestPermissionListener = onRequestPermissionListener
            var shouldShowDialog = false
            var permissionDialogStr = ""

            for ((key, value) in permissionsMap) {
                if (ContextCompat.checkSelfPermission(this@CloudLoginActivity, key) != PackageManager.PERMISSION_GRANTED) {
                    mPermissionToRequest.add(key)
                    permissionDialogStr += "$value，"
                    if (ActivityCompat.shouldShowRequestPermissionRationale(this@CloudLoginActivity, key)) {
                        shouldShowDialog = true
                    }
                }
            }

            if (permissionDialogStr.isNotEmpty()) {
                permissionDialogStr = permissionDialogStr.substring(0, permissionDialogStr.length - 1)
            }

            if (mPermissionToRequest.isNotEmpty()) {
                val names = mPermissionToRequest.toTypedArray()
                if (shouldShowDialog) {
                    showExplanation(getString(R.string.wecast_apply_permission),
                            String.format(getString(R.string.wecast_apply_relative_permission), permissionDialogStr),
                            names,
                            PERMISSION_REQUEST_CODE)
                } else {
                    mRequestPermissionSuccess = false
                    mIsShowRequestPermissionDialog = true
                    ActivityCompat.requestPermissions(this@CloudLoginActivity, names, PERMISSION_REQUEST_CODE)
                }
            } else {
                onRequestPermissionListener.onRequestSuccess()
            }
        } else {
            onRequestPermissionListener.onRequestSuccess()
        }
    }

    /**
     * 自动登录，同时保证闪屏最少显示2s
     */
    protected fun login() {
        if (TextUtils.isEmpty(WeCastInfo.instance.authCode)) {
            return
        }
        if (mLoginFinishFlag) {
            mLoginFinishFlag = false
            var nickName = intent.getStringExtra(DEVICE_NAME_KEY)
            if (TextUtils.isEmpty(nickName)) {
                nickName = DeviceInfoUtil.getDeviceName()
            } else {
                DeviceInfoUtil.setDeviceName(nickName)
            }
            CloudUtil.JniMethod.login(nickName)
            mBaseHandler?.postDelayed({
                mDelayFinishFlag = true
                tryJump()
            }, mSplashShowMinTime.toLong())
        }
    }

    /**
     * 申请权限显示说明
     *
     * @param title
     * @param message
     * @param permissions
     * @param permissionRequestCode
     */
    private fun showExplanation(title: String,
                                message: String,
                                permissions: Array<String>,
                                permissionRequestCode: Int) {
        mRequestPermissionSuccess = false
        showAlertDialog(title, message, getString(android.R.string.ok), {
            ActivityCompat.requestPermissions(this@CloudLoginActivity, permissions, permissionRequestCode)
            mRequestPermissionSuccess = true
            mIsShowRequestPermissionDialog = true
        }, null, null)
    }

    override fun onDestroy() {
        super.onDestroy()
        EventBusUtil.unregister(this)
        mFatalDialog?.dismiss()
        mFatalDialog = null
        mDownloadDialog?.dismiss()
        mDownloadDialog = null
    }

    override fun onRequestPermissionsResult(requestCode: Int, permissions: Array<String>, grantResults: IntArray) {
        when (requestCode) {
            PERMISSION_REQUEST_CODE -> {
                mIsShowRequestPermissionDialog = false
                var allPermissionGranted = true
                var permissionNotGrantedStr = ""
                if (permissions.isNotEmpty() && grantResults.isNotEmpty() && permissions.size == grantResults.size) {
                    for (i in grantResults.indices) {
                        if (grantResults[i] != PackageManager.PERMISSION_GRANTED) {
                            allPermissionGranted = false
                            permissionNotGrantedStr += mPermissionMap?.get(permissions[i]) + "，"
                        }
                    }
                } else {
                    allPermissionGranted = false
                }

                if (permissionNotGrantedStr.isNotEmpty()) {
                    permissionNotGrantedStr = permissionNotGrantedStr.substring(0, permissionNotGrantedStr.length - 1)
                }

                if (allPermissionGranted) {
                    mRequestPermissionSuccess = true
                    mOnRequestPermissionListener?.onRequestSuccess()
                } else {
                    Toast.makeText(this,
                            String.format(getString(R.string.wecast_apply_permission_failed), permissionNotGrantedStr),
                            Toast.LENGTH_SHORT).show()

                    onPermissionNotGranted()
                }
            }
            else -> {
                super.onRequestPermissionsResult(requestCode, permissions, grantResults)
            }
        }
    }

    open fun onPermissionNotGranted() {
        mBaseHandler?.postDelayed({ finish() }, 3000)
    }

    /**
     * 根据登录结果跳转
     */
    protected fun tryJump() {
        if (mDelayFinishFlag && mLoginFinishFlag) {
            if (mLoginResult && !mNeedUpdate && (mRequestPermissionSuccess == null || mRequestPermissionSuccess == true)) {
                jump()
            } else {
                if (!mIsShowRequestPermissionDialog) {
                    if (mDownloadDialog != null && mDownloadDialog!!.isShowing) {
                        return
                    }
                    runOnUiThread {
                        showFatalDialog(mLoginInfo, getString(R.string.wecast_start_cast_failed)) {
                            mFatalDialog?.dismiss()
                            login()
                        }
                    }
                }
            }
        }
    }

    abstract fun jump()

    protected fun showDownloadDialog(onDownloadCancelListener: DownLoadDialog.OnDownloadCancelListener?) {
        mDownloadDialog = DownLoadDialog(this@CloudLoginActivity)
        mDownloadDialog?.show()
        mDownloadDialog?.setOnDownloadCancelListener(onDownloadCancelListener)
        DownloadApkUtils.setDownloadProgressListener(this@CloudLoginActivity
                , object : DownloadApkUtils.WeCastDownloadApkProgressListener {
            override fun onProgressChange(progress: Int) {
                mDownloadDialog?.setProgress(progress)
            }
        })
    }

    /**
     * 严重错误对话框
     *
     * @param info
     */
    abstract fun showFatalDialog(info: String?, message: String, retryFunc: () -> Unit)

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onLoginSuccessEvent(event: SenderEvents.ELoginSuccess) {
        hideLoading()
        Logger.t(TAG).d("onLoginSuccessEvent")
        mLoginFinishFlag = true
        mLoginResult = true
        mReturnCode = 0
        mLoginInfo = "login success"
        tryJump()
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onLoginFailedEvent(event: SenderEvents.ELoginFailed) {
        hideLoading()
        Logger.t(TAG).d("onLoginFailedEvent errorCode:${event.errorCode}")
        if (event.errorCode == 15) { // 封版本
            return
        }
        mLoginFinishFlag = true
        mLoginResult = false
        mReturnCode = -1
        mLoginInfo = "login faWiled"
        tryJump()
    }

    @Subscribe(sticky = true, threadMode = ThreadMode.MAIN)
    fun onConfigUpdateEvent(event: SenderEvents.EUIConfigUpdate) {
        Logger.t(TAG).d("onConfigUpdateEvent castScheme = ${event.castScheme} ui:${event.uiConfig}")
        // 发送端xcast的使用是接口传参决定
        WeCastConfigManager.getInstance().setSchemeConfig(event.castScheme, false, true)
        WeCastUIConfigManager.getInstance().updateConfig(event.uiConfig)
        EventBusUtil.removeStick(event)
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    open fun onShowErrorEvent(event: SenderEvents.EShowError) {
        Logger.t(TAG).d("onShowErrorEvent errorCode = ${event.errorCode}")
        if (event.errorCode == 15) { // 封版本
            return
        }
        mLoginFinishFlag = true
        mLoginResult = false
        mReturnCode = -1
        mLoginInfo = "login failed"
        tryJump()
        EventBusUtil.removeStick(event)
    }

    @Subscribe(sticky = true, threadMode = ThreadMode.MAIN)
    fun onStopSplashEvent(event: SenderEvents.EStopSplashEvent) {
        finish()
        EventBusUtil.removeStick(event)
    }
}