package com.tencent.wecast.sender.cloud.utils

import android.app.Activity
import android.content.Context
import android.graphics.Rect
import android.view.View
import android.view.inputmethod.InputMethodManager

object KeyboardUtils {

    private var sContentViewInvisibleHeightPre: Int = 0

    /**
     * Show the soft input.
     *
     * @param activity The activity.
     */
    fun showSoftInput(activity: Activity) {
        val imm = activity.getSystemService(Activity.INPUT_METHOD_SERVICE) as InputMethodManager ?: return
        var view = activity.currentFocus
        if (view == null) view = View(activity)
        imm.showSoftInput(view, InputMethodManager.SHOW_FORCED)
    }

    /**
     * Hide the soft input.
     *
     * @param activity The activity.
     */
    fun hideSoftInput(activity: Activity) {
        val imm = activity.getSystemService(Activity.INPUT_METHOD_SERVICE) as InputMethodManager ?: return
        var view = activity.currentFocus
        if (view == null) view = View(activity)
        imm.hideSoftInputFromWindow(view.windowToken, 0)
    }


    /**
     * Toggle the soft input display or not.
     */
    fun toggleSoftInput(context: Context) {
        val imm = context.getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager ?: return
        imm.toggleSoftInput(InputMethodManager.SHOW_FORCED, 0)
    }

    /**
     * Return whether soft input is visible.
     *
     * @param activity             The activity.
     * @param minHeightOfSoftInput The minimum height of soft input.
     * @return `true`: yes<br></br>`false`: no
     */
    @JvmOverloads
    fun isSoftInputVisible(activity: Activity, minHeightOfSoftInput: Int = 200): Boolean {
        return getContentViewInvisibleHeight(activity) >= minHeightOfSoftInput
    }

    private fun getContentViewInvisibleHeight(activity: Activity): Int {
        val contentView = activity.findViewById<View>(android.R.id.content)
        val outRect = Rect()
        contentView.getWindowVisibleDisplayFrame(outRect)
        return contentView.bottom - outRect.bottom
    }

    /**
     * Register soft input changed listener.
     *
     * @param activity The activity.
     * @param listener The soft input changed listener.
     */
    fun registerSoftInputChangedListener(activity: Activity,
                                         listener: OnSoftInputChangedListener?) {
        val contentView = activity.findViewById<View>(android.R.id.content)
        sContentViewInvisibleHeightPre = getContentViewInvisibleHeight(activity)
        contentView.viewTreeObserver.addOnGlobalLayoutListener {
            if (listener != null) {
                val height = getContentViewInvisibleHeight(activity)
                if (sContentViewInvisibleHeightPre != height) {
                    listener.onSoftInputChanged(height)
                    sContentViewInvisibleHeightPre = height
                }
            }
        }
    }

    /**
     * Fix the leaks of soft input.
     *
     * Call the function in [Activity.onDestroy].
     *
     * @param context The context.
     */
    fun fixSoftInputLeaks(context: Context?) {
        if (context == null) return
        val imm = context.getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager ?: return
        val strArr = arrayOf("mCurRootView", "mServedView", "mNextServedView")
        for (i in 0..2) {
            try {
                val declaredField = imm.javaClass.getDeclaredField(strArr[i]) ?: continue
                if (!declaredField.isAccessible) {
                    declaredField.isAccessible = true
                }
                val obj = declaredField.get(imm)
                if (obj == null || obj !is View) continue
                if (obj.context === context) {
                    declaredField.set(imm, null)
                } else {
                    return
                }
            } catch (th: Throwable) {
                th.printStackTrace()
            }

        }
    }


    interface OnSoftInputChangedListener {
        fun onSoftInputChanged(height: Int)
    }
}
