package com.tencent.wecast;

import android.content.Context;
import android.content.SharedPreferences;

import com.google.gson.JsonObject;
import com.tencent.wecast.utils.Logger;
import com.tencent.wecast.utils.StorageUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

// 投屏功能策略等配置项的管理类
public class WeCastConfigManager {
    public final static int CAST_SCHEME_LAN_NET = 1;       // 局域网方案
    public final static int CAST_SCHEME_PUBLIC_NET = 2;    // 公网方案

    public final static int CLOUD_TYPE_OPEN = 0;    // 公有云类型
    public final static int CLOUD_TYPE_PRIVATE = 1;    // 私有云类型

    public final static int TYPE_RECEIVER = 0;
    public final static int TYPE_SENDER = 0;
    private static final WeCastConfigManager ourInstance = new WeCastConfigManager();

    public static WeCastConfigManager getInstance() {
        return ourInstance;
    }

    private WeCastConfigManager() {
    }

    private FeatureTag mFeatureTag = new FeatureTag();
    private Context mAppContext;

    public void init(Context context, int type) {
        mAppContext = context.getApplicationContext();
        SharedPreferences sp = mAppContext.getSharedPreferences(getCloudConfigSpName(),
                Context.MODE_PRIVATE);
        mFeatureTag.cloudType = sp.getInt("cloud_type", CLOUD_TYPE_OPEN);
        mFeatureTag.cloudCenterUrl = sp.getString("config_center_url", "");
        mFeatureTag.castScheme = sp.getInt("cast_scheme", mFeatureTag.castScheme);
        mFeatureTag.isOpenAirPlay = sp.getBoolean("open_airplay", mFeatureTag.isOpenAirPlay);
        if (type == TYPE_RECEIVER) {
            save4Launcher();
        }
    }

    private String getCloudConfigSpName() {
        return "wecast_cloud_config" + WeCastEnv.getCurrentEnv();
    }

    // 存储到sdcard给Launcher使用
    private void save4Launcher() {
        final String cloudFilePath = StorageUtil.getCloudConfigPath(mAppContext)
                + File.separatorChar + "cloud_type_info.json";
        JsonObject outObj = new JsonObject();
        outObj.addProperty("cloud_type", mFeatureTag.cloudType);
        outObj.addProperty("config_center_url", mFeatureTag.cloudCenterUrl);
        FileOutputStream fos = null;
        try {
            File cloudFile = new File(cloudFilePath);
            if (!cloudFile.exists()) {
                cloudFile.createNewFile();
            }
            fos = new FileOutputStream(cloudFile);
            fos.write(outObj.toString().getBytes());
            fos.flush();
        } catch (IOException e) {
            Logger.kp().e("Write cloud config to file failure: " + e.getMessage());
        } finally {
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    public int getAppType() {
        if (mFeatureTag.cloudType == CLOUD_TYPE_OPEN) {
            return WeCastConstant.WeCastAppType.TYPE_WECASTOPEN;
        } else if (mFeatureTag.cloudType == CLOUD_TYPE_PRIVATE) {
            return WeCastConstant.WeCastAppType.TYPE_WECASTPRIVATE;
        }
        return WeCastConstant.WeCastAppType.TYPE_WECASTOPEN;
    }

    public int getCloudType() {
        return mFeatureTag.cloudType;
    }

    public String getConfigUrl() {
        return mFeatureTag.cloudCenterUrl;
    }

    public void setSchemeConfig(int castScheme, boolean openAirplay) {
        mFeatureTag.castScheme = castScheme;
        mFeatureTag.isOpenAirPlay = openAirplay;
        SharedPreferences.Editor edit = mAppContext.getSharedPreferences(getCloudConfigSpName(),
                Context.MODE_PRIVATE).edit();
        edit.putInt("cast_scheme", mFeatureTag.castScheme)
                .putBoolean("open_airplay", mFeatureTag.isOpenAirPlay)
                .apply();
    }

    public void setSchemeConfig(int castScheme) {
        setSchemeConfig(castScheme, false);
    }

    // 是否是局域网方案（动态更改的）
    public boolean isLanScheme() {
        return mFeatureTag.castScheme == CAST_SCHEME_LAN_NET;
    }

    // 是否使用AirPlay服务
    public boolean isOpenAirPlay() {
        return mFeatureTag.isOpenAirPlay;
    }

    private static class FeatureTag {
        public int cloudType = CLOUD_TYPE_OPEN;
        public String cloudCenterUrl = "";
        public int castScheme = CAST_SCHEME_PUBLIC_NET;
        public boolean isOpenAirPlay = false;   // 是否打开airplay服务
    }

    public String tipsInfo() {
        return info();
    }

    public String info() {
        return String.format("type:%s|url:%s|scheme:%s|airplay:%s",
                mFeatureTag.cloudType == 0 ? "open" : "private",
                mFeatureTag.cloudCenterUrl,
                mFeatureTag.castScheme == CAST_SCHEME_PUBLIC_NET ? "wan" : "wLan",
                mFeatureTag.isOpenAirPlay ? "true" : "false");
    }

    public boolean saveCenterConfig(int cloudType, String centerUrl) {
        if (mFeatureTag.cloudType == cloudType
                && mFeatureTag.cloudCenterUrl.equals(centerUrl)) {
            return true;
        }

        mFeatureTag.cloudType = cloudType;
        mFeatureTag.cloudCenterUrl = centerUrl;
        SharedPreferences.Editor edit = mAppContext.getSharedPreferences(getCloudConfigSpName(),
                Context.MODE_PRIVATE).edit();
        edit.putInt("cloud_type", mFeatureTag.cloudType)
                .putString("config_center_url", mFeatureTag.cloudCenterUrl)
                .apply();
        save4Launcher();
        return false;
    }
}
