package com.tencent.tiw.logger.http;

import android.util.Log;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URL;
import java.net.UnknownHostException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

public class     HttpClient {
    private static final String TAG = HttpClient.class.getSimpleName();

    public static final int HTTP_ACTION_REQUEST = 0;
    public static final int HTTP_ACTION_RESPONSE = 1;

    //获取当前设备的CPU数
    private static final int CPU_COUNT = Runtime.getRuntime().availableProcessors();
    //核心池大小设为CPU数加1
    private static final int CORE_POOL_SIZE = CPU_COUNT + 1;
    //设置线程池的最大大小
    private static final int MAX_POOL_SIZE = 2 * CPU_COUNT + 1;
    //存活时间
    private static final long KEEP_ALIVE = 5L;

    //创建证书管理器
    private static TrustManager[] mTrustManagers = new TrustManager[]{
            new X509TrustManager() {
                public X509Certificate[] getAcceptedIssuers() {
                    return new X509Certificate[]{};
                }

                public void checkClientTrusted(X509Certificate[] chain, String authType) {
                    try {
                        chain[0].checkValidity();
                    } catch (CertificateExpiredException e) {
                        e.printStackTrace();
                    } catch (CertificateNotYetValidException e) {
                        e.printStackTrace();
                    }
                }

                public void checkServerTrusted(X509Certificate[] chain, String authType) {
                    try {
                        chain[0].checkValidity();
                    } catch (CertificateExpiredException e) {
                        e.printStackTrace();
                    } catch (CertificateNotYetValidException e) {
                        e.printStackTrace();
                    }
                }
            }
    };

    //创建域名过滤器
    private static HostnameVerifier mHostnameVerifier = new HostnameVerifier() {
        @Override
        public boolean verify(String hostname, SSLSession session) {
            if (hostname.contains("cloud") || hostname.contains("tim")) { // Android平台：C++调用JNI到HttpClient,这里verify是对https做校验
                return true;
            }
            return false;
        }
    };

    //创建线程池对象
    private static final Executor mThreadPoolExecutor = new ThreadPoolExecutor(CORE_POOL_SIZE, MAX_POOL_SIZE, KEEP_ALIVE, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());

    private static native void nativeProgressCallback(int action, int currentSize, int totalSize, long callback);
    private static native void nativeResponseCallback(int code, Map<String, String> headers, byte[] response, long progressCallback, long responseCallback);

    public interface HttpRequestListener {
        void onProgress(final int action, final int currentSize, final int totalSize);

        void onCompleted(final int code, final Map<String, String> headers, final byte[] response);
    }

    private static void httpRequest(final String method, final String url,
                                     final Map<String, String> headers,
                                     final byte[] content,
                                     final String uploadFile,
                                     final String downloadFile,
                                     final HttpRequestListener listener,
                                     final String proxyHost, final int proxyPort,
                                     final int connectTimeout, final int recvTimeout) {
        //创建一个新的请求任务
        Runnable requestRunnable = new Runnable() {
            @Override
            public void run() {
                HttpURLConnection conn = null;
                InputStream responseInputStream = null;
                int code = HttpURLConnection.HTTP_OK;
                Map<String, String> rspHeaders = new HashMap<String, String>();
                byte[] response = null;

                try {
                    if (!proxyHost.isEmpty() && proxyPort != 0) {
                        Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));
                        conn = (HttpURLConnection) (new URL(url).openConnection(proxy));
                    } else {
                        conn = (HttpURLConnection) (new URL(url).openConnection());
                    }
                    boolean useHttps = url.startsWith("https");
                    if (useHttps) {
                        SSLContext sslContext = SSLContext.getInstance("TLS");
                        sslContext.init(null, mTrustManagers, null);

                        HttpsURLConnection httpsConn = (HttpsURLConnection) conn;
                        httpsConn.setSSLSocketFactory(sslContext.getSocketFactory());
                        httpsConn.setHostnameVerifier(mHostnameVerifier);
                    }
                    conn.setRequestMethod(method);
                    conn.setConnectTimeout(connectTimeout);
                    conn.setReadTimeout(recvTimeout);
                    conn.setUseCaches(false);
                    conn.setDoInput(true);

                    if (headers != null) {
                        for (Map.Entry<String, String> entry : headers.entrySet()) {
                            conn.setRequestProperty(entry.getKey(), entry.getValue());
                        }
                    }

                    //写入请求数据
                    boolean hasContent = content != null && content.length > 0;
                    boolean hasUploadFile = uploadFile != null && uploadFile.length() != 0;
                    boolean hasOutput = hasContent || hasUploadFile;
                    if (!method.equalsIgnoreCase("GET") && hasOutput) {
                        conn.setDoOutput(true);
                        InputStream inputStream = null;
                        if (hasUploadFile) {
                            inputStream = new FileInputStream(uploadFile);
                        } else {
                            inputStream = new ByteArrayInputStream(content);
                        }
                        if (inputStream != null) {
                            int totalSize = inputStream.available(); //TODO alderzhang：此处大于2G的文件会出现问题
                            conn.setFixedLengthStreamingMode(totalSize);
                            int bytesTransferred = 0;
                            byte[] buf = new byte[4096];
                            while (true) {
                                int len = inputStream.read(buf);
                                if (len < 0) break;
                                bytesTransferred += len;
                                conn.getOutputStream().write(buf, 0, len);
                                if (listener != null) {
                                    listener.onProgress(HTTP_ACTION_REQUEST, bytesTransferred, totalSize);
                                }
                            }
                            inputStream.close();
                        }
                    }

                    code = conn.getResponseCode();

                    int headerCount = conn.getHeaderFields().size();
                    if (headerCount > 0) {
                        for (int i = 0; i < headerCount; ++i) {
                            String key = conn.getHeaderFieldKey(i);
                            if (key == null) {
                                continue;
                            }
                            String value = conn.getHeaderField(i);
                            rspHeaders.put(key, value);
                        }
                    }

                    int totalSize = conn.getContentLength();
                    if (totalSize == -1) {
                        totalSize = 0;
                    }

                    //读取返回数据
                    boolean hasDownloadFile = downloadFile != null && downloadFile.length() != 0;
                    if (code == HttpURLConnection.HTTP_OK) {
                        responseInputStream = new BufferedInputStream(conn.getInputStream());
                        OutputStream outputStream = null;
                        if (hasDownloadFile) {
                            outputStream = new FileOutputStream(downloadFile);
                        } else {
                            outputStream = new ByteArrayOutputStream();
                        }
                        if (outputStream != null) {
                            int bytesReceived = 0;
                            byte[] buf = new byte[4096];
                            while (true) {
                                int len = responseInputStream.read(buf);
                                if (len < 0) break;
                                bytesReceived += len;
                                outputStream.write(buf, 0, len);
                                if (listener != null) {
                                    listener.onProgress(HTTP_ACTION_RESPONSE, bytesReceived, totalSize);
                                }
                            }
                            if (hasDownloadFile) {
                                response = new byte[]{};
                            } else {
                                response = ((ByteArrayOutputStream) outputStream).toByteArray();
                            }
                            outputStream.close();
                        }
                    }

                } catch (UnknownHostException e) {
                    code = HttpURLConnection.HTTP_NOT_FOUND;
                } catch (Exception e) {
                    code = -1;
                    response = Log.getStackTraceString(e).getBytes();
                } finally {
                    //清理资源
                    if (responseInputStream != null) {
                        try {
                            responseInputStream.close();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    }

                    if (conn != null) {
                        conn.disconnect();
                    }

                    if (listener != null) {
                        listener.onCompleted(code, rspHeaders, response);
                    }
                }
            }
        };
        mThreadPoolExecutor.execute(requestRunnable);
    }

    public static void httpRequest(final String method,
                                    final String url,
                                    final Map<String, String> headers,
                                    final byte[] content,
                                    final String uploadFile,
                                    final String downloadFile,
                                    final long progressCallback,
                                    final long responseCallback,
                                    final String proxyHost, final int proxyPort) {
//        Map<String, String> headers = null;
//        if (headerKeys != null && headerValues != null && headerKeys.length == headerValues.length) {
//            headers = new HashMap<String, String>();
//            for (int i = 0; i < headerKeys.length; ++i) {
//                headers.put(headerKeys[i], headerValues[i]);
//            }
//        }
        httpRequest(method, url, headers, content, uploadFile, downloadFile, new HttpRequestListener() {
            @Override
            public void onProgress(int action, int currentSize, int totalSize) {
                if (progressCallback != 0) {
                    try {
                        nativeProgressCallback(action, currentSize, totalSize, progressCallback);
                    } catch(UnsatisfiedLinkError e) {
                        // 云集反馈: 只有Android4.4.2上会出现该错误
                    }
                }
            }

            @Override
            public void onCompleted(int code, Map<String, String> headers, byte[] response) {
                if (responseCallback != 0) {
                    try {
                        nativeResponseCallback(code, headers, response, progressCallback, responseCallback);
                    } catch(UnsatisfiedLinkError e) {
                        // 云集反馈: 只有Android4.4.2上会出现该错误
                        Log.i(TAG, "onCompleted: ");
                    }
                }
            }
        }, proxyHost, proxyPort, 5000, 10000);
    }
}
