package com.taboola.android.plus;

import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.Keep;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.support.annotation.UiThread;
import android.support.annotation.WorkerThread;
import android.util.Log;

import com.taboola.android.SdkCore;
import com.taboola.android.api.TaboolaApi;
import com.taboola.android.plus.content.ContentConfig;
import com.taboola.android.plus.content.LanguagesConfig;
import com.taboola.android.plus.content.TBContentManager;
import com.taboola.android.plus.notification.BridgeInternal;
import com.taboola.android.plus.notification.NotificationConfig;
import com.taboola.android.plus.notification.TBNotificationAnalyticsManager;
import com.taboola.android.plus.notification.TBNotificationLocalStore;
import com.taboola.android.plus.notification.TBNotificationManager;
import com.taboola.android.plus.notification.killSwitch.FrequentCrashBlockConfig;
import com.taboola.android.plus.notification.killSwitch.SdkPlusExceptionHandler;

import java.util.HashMap;
import java.util.Map;

@Keep
@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class TaboolaPlus implements PublicApi.PublicTaboolaPlus {
    private static final String TAG = TaboolaPlus.class.getSimpleName();
    private static TaboolaPlus singleton = null;
    private static Context applicationContext;
    private static SdkCore sdkCore;

    private SdkPlusExceptionHandler sdkPlusExceptionHandler;

    private boolean isInitialized = false;

    private TBNotificationManager notificationManager;
    private TBNotificationLocalStore notificationLocalStorage;

    private TaboolaPlus() {
        notificationLocalStorage = new TBNotificationLocalStore(applicationContext);
        notificationManager = com.taboola.android.plus.notification.BridgeInternal
                .newTBNotificationManager(applicationContext,
                        new TBContentManager(applicationContext), notificationLocalStorage);
        initSdkPlusUncaughtExceptionHandler();
    }

    private static TaboolaPlus getInstanceInternal() {
        if (singleton == null) {
            synchronized (TaboolaPlus.class) {
                if (singleton == null) {
                    singleton = new TaboolaPlus();
                }
            }
        }
        return singleton;
    }

    static void setApplicationContext(@NonNull Context applicationContext) {
        TaboolaPlus.applicationContext = applicationContext;
        if (applicationContext == null) {
            Log.e(TAG, "setApplicationContext: context is null");
        } else {
            Log.d(TAG, "setApplicationContext: context is not null");
        }
    }

    public static void setSdkCore(SdkCore sdkCore) {
        TaboolaPlus.sdkCore = sdkCore;
    }

    @Nullable
    static Context getApplicationContext() { // todo refactor
        return applicationContext;
    }

    /**
     * Must be called in order to use TaboolaPlus object (Should be called only once). Requires
     * internet connection on first launch (after that cache can be used). After either successful or
     * failed init, corresponding callback will be called. If you also want to be notified when init has failed use
     * {@link TaboolaPlus#init(String, String, Map, TaboolaPlusRetrievedCallback, TaboolaPlusRetrieveFailedCallback)}
     *
     * @param publisherName        publisherName provided by your Taboola account manager
     * @param configId             configId provided by your Taboola account manager
     * @param onSuccessfulCallback callback that will provide initialized TaboolaPlus object
     */
    @UiThread
    public static void init(@NonNull String publisherName,
                            @NonNull String configId,
                            @NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback) {
        init(publisherName, configId, onSuccessfulCallback, null);
    }

    /**
     * Must be called in order to use TaboolaPlus object (Should be called only once). Requires
     * internet connection on first launch (after that cache can be used). After either successful or
     * failed init, corresponding callback will be called.
     *
     * @param publisherName        publisherName provided by your Taboola account manager
     * @param configId             configId provided by your Taboola account manager
     * @param onSuccessfulCallback callback that will provide initialized TaboolaPlus object
     * @param onFailedCallback     callback that will provide {@code Throwable} that indicates init error
     */
    @UiThread
    public static void init(@NonNull String publisherName,
                            @NonNull String configId,
                            @NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback,
                            @Nullable final TaboolaPlusRetrieveFailedCallback onFailedCallback) {
        init(publisherName, configId, null, onSuccessfulCallback, onFailedCallback);
    }


    /**
     * Must be called in order to use TaboolaPlus object (Should be called only once). Requires
     * internet connection on first launch (after that cache can be used). After either successful or
     * failed init, corresponding callback will be called.
     *
     * @param publisherName              publisherName provided by your Taboola account manager
     * @param configId                   configId provided by your Taboola account manager
     * @param taboolaPlusExtraProperties publisher specific properties. Only set them if you were explicitly told to do so
     * @param onSuccessfulCallback       callback that will provide initialized TaboolaPlus object
     * @param onFailedCallback           callback that will provide {@code Throwable} that indicates init error
     */
    @UiThread
    public static void init(@NonNull String publisherName,
                            @NonNull String configId,
                            @Nullable Map<String, String> taboolaPlusExtraProperties,
                            @NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback,
                            @Nullable final TaboolaPlusRetrieveFailedCallback onFailedCallback) {
        final TaboolaPlus taboolaPlus = getInstanceInternal();
        TBNotificationLocalStore localStorage = taboolaPlus.notificationLocalStorage;

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            Log.i(TAG, "isInitialized: Android version is less than LOLLIPOP. Ignoring call.");
            onSuccessfulCallback.onTaboolaPlusRetrieved(taboolaPlus);
            return;
        }

        setDefaultSettings(localStorage);

        localStorage.setPublisher(publisherName);
        localStorage.setConfigId(configId);
        localStorage.setTaboolaPlusExtraProperties(taboolaPlusExtraProperties);

        applyNewConfigs(publisherName, configId, taboolaPlusExtraProperties, onSuccessfulCallback, onFailedCallback, taboolaPlus);

        ConfigManager.triggerAsyncConfigsUpdate(applicationContext, publisherName, configId);
    }

    private static void setDefaultSettings(@NonNull TBNotificationLocalStore localStore) {
        localStore.setWifiOnlyModeRuntimeFlag(false);
        localStore.setEnabled(false);
        localStore.setApplicationName("");
        localStore.setTaboolaPlusExtraProperties(null);
    }


    @WorkerThread // todo move out of UI thread
    private void initSdkPlusUncaughtExceptionHandler() {

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            Log.v(TAG, "initSdkPlusUncaughtExceptionHandler: Android version is less than LOLLIPOP. Ignoring call.");
            return;
        }

        ConfigManager.CachedSdkConfigContainer currentSdkConfig =
                ConfigManager.getCurrentSdkConfig(applicationContext);

        if (currentSdkConfig == null) {
            Log.v(TAG, "initSdkPlusUncaughtExceptionHandler: current sdkPlus config is empty skipping ExceptionHandler init");
        } else {
            FrequentCrashBlockConfig frequentCrashBlockConfig = currentSdkConfig
                    .getSdkPlusConfig()
                    .getNotificationConfig()
                    .getKillSwitchConfig().getFrequentCrashBlockConfig();

            if (frequentCrashBlockConfig.isFrequentCrashBlockEnabled() && sdkPlusExceptionHandler == null) {
                Map<String, String> map = new HashMap<>();
                map.put("setGUEH", String.valueOf(true));
                sdkCore.setExtraProperties(map);

                sdkPlusExceptionHandler =
                        new SdkPlusExceptionHandler(applicationContext, sdkCore.getNetworkManager(),
                                frequentCrashBlockConfig);

                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        //register sdk plus exception handler with sdk core
                        sdkCore.registerTaboolaExceptionHandler(sdkPlusExceptionHandler);
                    }
                });
            }
        }
    }

    @Override
    public TBNotificationManager getNotificationManager() {
        return notificationManager;
    }

    /**
     * Restore TaboolaPlus with last used init parameters but with new(!) config, or
     * reInit existing instance with new config
     */
    @UiThread
    static void reInit(@NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback,
                       @Nullable final TaboolaPlusRetrieveFailedCallback onFailedCallback) {
        final TaboolaPlus taboolaPlus = TaboolaPlus.getInstanceInternal();
        String publisherName = taboolaPlus.notificationLocalStorage.getPublisher();
        String configId = taboolaPlus.notificationLocalStorage.getConfigId();
        @Nullable HashMap<String, String> extraProperties =
                taboolaPlus.notificationLocalStorage.getTaboolaPlusExtraProperties();

        applyNewConfigs(publisherName, configId, extraProperties, onSuccessfulCallback,
                onFailedCallback, taboolaPlus);

        ConfigManager.triggerAsyncConfigsUpdate(applicationContext, publisherName, configId);
    }

    @UiThread
    private static void applyNewConfigs(@NonNull String publisherName, @NonNull String configId,
                                        @Nullable final Map<String, String> taboolaPlusExtraProperties,
                                        @NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback,
                                        @Nullable final TaboolaPlusRetrieveFailedCallback onFailedCallback,
                                        @NonNull final TaboolaPlus taboolaPlus) {
        ConfigManager.getNewConfigs(applicationContext, publisherName, configId, true,
                new ConfigManager.OnGetAllConfigsCallback() {
                    @Override
                    public void onConfigsFetched(SdkPlusConfig config, LanguagesConfig languagesConfig) {
                        TBNotificationLocalStore localStore = new TBNotificationLocalStore(applicationContext);
                        TBNotificationAnalyticsManager analyticsManager = new TBNotificationAnalyticsManager(applicationContext, localStore);
                        analyticsManager.sendNotificationConfigRefreshedEvent();

                        taboolaPlus.applyConfigs(config, languagesConfig, taboolaPlusExtraProperties, onSuccessfulCallback);
                    }

                    @Override
                    public void onConfigsFailed(Throwable throwable) {
                        Log.e(TAG, "applyNewConfigs: failed to get configs: " + throwable.getMessage());

                        if (onFailedCallback != null) {
                            onFailedCallback.onTaboolaPlusRetrieveFailed(throwable);
                        }
                    }
                });
    }

    /**
     * Restore TaboolaPlus with last used init parameters and config
     */
    @UiThread
    static void restore(@NonNull final TaboolaPlusRetrievedCallback successfulCallback,
                        @Nullable final TaboolaPlusRetrieveFailedCallback failedCallback) {
        final TaboolaPlus taboolaPlus = TaboolaPlus.getInstanceInternal();

        if (taboolaPlus.isInitialized() && TaboolaApi.getInstance().isInitialized()) {
            Log.i(TAG, "restore: successful (already initialized)");
            successfulCallback.onTaboolaPlusRetrieved(taboolaPlus);
        } else {
            String publisherName = taboolaPlus.notificationLocalStorage.getPublisher();
            String configId = taboolaPlus.notificationLocalStorage.getConfigId();
            @Nullable final HashMap<String, String> extraProperties =
                    taboolaPlus.notificationLocalStorage.getTaboolaPlusExtraProperties();

            ConfigManager.getCurrentConfigs(applicationContext, publisherName, configId,
                    new ConfigManager.OnGetAllConfigsCallback() {
                        @Override
                        public void onConfigsFetched(SdkPlusConfig config, LanguagesConfig languagesConfig) {
                            taboolaPlus.applyConfigs(config, languagesConfig, extraProperties, successfulCallback);
                        }

                        @Override
                        public void onConfigsFailed(Throwable throwable) {
                            Log.e(TAG, "restore: failed to get configs: " + throwable.getMessage());
                            if (failedCallback != null) {
                                failedCallback.onTaboolaPlusRetrieveFailed(throwable);
                            }
                        }
                    });
        }
    }

    @UiThread
    private void applyConfigs(@NonNull SdkPlusConfig sdkPlusConfig,
                              @NonNull LanguagesConfig languagesConfig,
                              @Nullable Map<String, String> taboolaPlusExtraProperties,
                              @NonNull TaboolaPlusRetrievedCallback successfulCallback) {
        NotificationConfig notificationConfig = sdkPlusConfig.getNotificationConfig();
        ContentConfig contentConfig = sdkPlusConfig.getContentConfig();

        notificationLocalStorage.setNonClickableUrlMarker(notificationConfig.getNonClickableUrlMarker());
        String configVariant = notificationConfig.getConfigVariant();
        if (configVariant != null) {
            notificationLocalStorage.setConfigVariant(configVariant);
        }

        initTaboolaApi(applicationContext, contentConfig, languagesConfig, taboolaPlusExtraProperties);

        BridgeInternal.applyConfig(notificationManager, notificationConfig, contentConfig);

        isInitialized = true;

        successfulCallback.onTaboolaPlusRetrieved(this);
    }

    @UiThread
    static void initTaboolaApi(Context appContext, @NonNull ContentConfig contentConfig,
                               @NonNull LanguagesConfig languagesConfig,
                               @Nullable Map<String, String> taboolaPlusExtraProperties) {
        String localizedPublisherName = ConfigManager
                .getLocalizedPublisherName(appContext, languagesConfig);
        String apiKey = contentConfig.getTaboolaApiConfig().getApiKey();

        Map<String, String> mergedExtraProperties = new HashMap<>();

        if (taboolaPlusExtraProperties != null) {
            // do not pass extra properties that are only used by TaboolaPlus
            mergedExtraProperties.putAll(taboolaPlusExtraProperties);
        }

        HashMap<String, String> taboolaApiExtraProperties = contentConfig.getTaboolaApiConfig().getTaboolaApiExtraProperties();
        if (taboolaApiExtraProperties != null) {
            mergedExtraProperties.putAll(taboolaApiExtraProperties);
        }

        mergedExtraProperties.put("enableFullRawDataResponse", "true");

        TaboolaApi.getInstance().init(appContext, localizedPublisherName, apiKey, mergedExtraProperties);
    }

    private boolean isInitialized() {
        return isInitialized;
    }

    public interface TaboolaPlusRetrievedCallback {

        void onTaboolaPlusRetrieved(TaboolaPlus taboolaPlus);
    }

    public interface TaboolaPlusRetrieveFailedCallback {

        void onTaboolaPlusRetrieveFailed(Throwable throwable);
    }
}