package com.taboola.android.infra.persistence;

import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.util.Log;

import androidx.annotation.NonNull;

import com.taboola.android.infra.utilities.Subscription;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

public abstract class AbstractPersistentResource<T> implements PersistentResource<T> {
    private static final String TAG = AbstractPersistentResource.class.getSimpleName();

    public interface Builder<T> {

        @NonNull
        T getDefaultValue();

        @NonNull
        String getStorageKey();

        @NonNull
        SharedPreferences getSharedPreferences();
    }

    protected final Builder<T> builder;

    private final Set<OnChangeListener<T>> onChangeListeners = new CopyOnWriteArraySet<>();

    private final SharedPreferences.OnSharedPreferenceChangeListener spListener = (sharedPreferences, key) -> {
        if (!getStorageKey().equals(key)) {
            return;
        }
//			Log.d(TAG, "onSharedPreferenceChanged() called with: onChangeListener = [" + onChangeListener + "], key = [" + key + "]");
        for (OnChangeListener<T> onChangeListener : onChangeListeners) {
            onChangeListener.onValueChanged(AbstractPersistentResource.this);
        }
    };

    AbstractPersistentResource(Builder<T> builder) {
        this.builder = builder;
    }

    @NonNull
    protected String getStorageKey() {
        return builder.getStorageKey();
    }

    protected SharedPreferences getSharedPreferences() {
        return builder.getSharedPreferences();
    }

    @NonNull
    @Override
    public T get() {
        setIfAbsent(builder.getDefaultValue());
        return getStoredValue();
    }

    @Override
    public void set(T value) {
        Editor edit = getSharedPreferences().edit();
        storeValue(edit, value);
        edit.apply();
    }

    @NonNull
    @Override
    public Subscription registerListener(OnChangeListener<T> onChangeListener) {
        boolean prevState = onChangeListeners.isEmpty();
        onChangeListeners.add(onChangeListener);
        boolean newState = onChangeListeners.isEmpty();
        if (prevState != newState) {
            ensureListenerState();
        }
        return () -> unregisterListener(onChangeListener);
    }

    private void ensureListenerState() {
        if (!onChangeListeners.isEmpty()) {
            getSharedPreferences().registerOnSharedPreferenceChangeListener(spListener);
        } else {
            getSharedPreferences().unregisterOnSharedPreferenceChangeListener(spListener);
        }
    }

    @Override
    public void unregisterListener(OnChangeListener<T> onChangeListener) {
        boolean prevState = onChangeListeners.isEmpty();
        onChangeListeners.remove(onChangeListener);
        boolean newState = onChangeListeners.isEmpty();
        if (prevState != newState) {
            ensureListenerState();
        }
    }

    @NonNull
    @Override
    public String toString() {
        return "[storageKey=" +
                getStorageKey() +
                ", builder=" +
                builder +
                ", get()=" +
                get() +
                "]";
    }

    protected abstract T getStoredValue();

    protected abstract void storeValue(Editor editor, T value);

    private void setIfAbsent(T value) {
        if (!isSet()) {
            Log.d(TAG, "setIfAbsent(" + getStorageKey() + ") - set: " + value);
            set(value);
        }
    }

    private boolean isSet() {
        return getSharedPreferences().contains(getStorageKey());
    }

}
