package com.taboola.android.infra.inappupdate.internals;

import androidx.annotation.AnyThread;
import androidx.core.util.Consumer;

import com.google.android.play.core.appupdate.AppUpdateManager;
import com.google.android.play.core.install.InstallStateUpdatedListener;
import com.google.android.play.core.install.model.InstallStatus;
import com.google.android.play.core.install.model.UpdateAvailability;
import com.taboola.android.infra.inappupdate.InAppUpdateException;
import com.taboola.android.infra.inappupdate.TriggerEventsCallback;
import com.taboola.android.infra.inappupdate.UpdateFlowException;
import com.taboola.android.infra.utilities.Executor2;

import java.util.Locale;

class UpdateInProgressMonitor implements Runnable {
    private final Executor2 uiExecutor;
    private final TriggerEventsCallback callback;
    private final AppUpdateManager updateManager;
    private final InstallStateUpdatedListener installStateUpdatedListener;

    UpdateInProgressMonitor(Executor2 uiExecutor, TriggerEventsCallback callback, AppUpdateManager updateManager) {
        this.uiExecutor = uiExecutor;
        this.callback = callback;
        this.updateManager = updateManager;
        installStateUpdatedListener = state -> {
            int status = state.installStatus();
            switch (status) {
                case InstallStatus.DOWNLOADING:
                    long bytesDownloaded = state.bytesDownloaded();
                    long totalBytesToDownload = state.totalBytesToDownload();
                    notifyCaller(it -> it.onDownloadProgress(bytesDownloaded, totalBytesToDownload));
                    break;
                case InstallStatus.DOWNLOADED:
                    notifyCaller(it -> it.onUpdateReadyToInstall(this.updateManager::completeUpdate));
                    onFinalState();
                    break;
                case InstallStatus.PENDING:
                    break;
                case InstallStatus.INSTALLING:
                    notifyCaller(TriggerEventsCallback::onInstalling);
                    onFinalState();
                    break;
                default:
                    //FAILED, CANCELLED, UNKNOWN
                    notifyCaller(it -> it.onUpdateFlowFailed(
                            new UpdateFlowException(
                                    String.format(Locale.US, "install monitor reported failure: state=%d code=%s", status, state.installErrorCode()),
                                    getErrorCode(status))));
                    onFinalState();
            }
            // Log state or install the update.
        };
    }

    private InAppUpdateException.ErrorCode getErrorCode(@InstallStatus int status) {
        if (status == InstallStatus.CANCELED) {
            return InAppUpdateException.ErrorCode.UPDATE_CANCELLED;
        }
        return InAppUpdateException.ErrorCode.UPDATE_FAILED;
    }

    @Override
    public void run() {
        updateManager
                .getAppUpdateInfo()
                .addOnSuccessListener(info -> {
                    int availability = info.updateAvailability();
                    if (availability != UpdateAvailability.DEVELOPER_TRIGGERED_UPDATE_IN_PROGRESS) {
                        notifyCaller(it -> it.onPreconditionsCheckFailed(
                                new UpdateFlowException("unexpected availability at this point: " + availability,
                                        InAppUpdateException.ErrorCode.ERROR_UNEXPECTED)));
                    }

                    updateManager.registerListener(installStateUpdatedListener);
                });
    }

    private void onFinalState() {
        updateManager.unregisterListener(installStateUpdatedListener);
    }

    @AnyThread
    private void notifyCaller(Consumer<TriggerEventsCallback> withCallback) {
        uiExecutor.submit(() -> withCallback.accept(callback));
    }


}
