package com.taboola.android.infra.inappupdate;

import androidx.annotation.IntRange;
import androidx.annotation.Keep;
import androidx.annotation.NonNull;

import com.google.android.play.core.install.model.AppUpdateType;
import com.taboola.android.infra.utilities.Validators;

import java.util.Arrays;

@Keep
public class Configuration {
    private final long targetVersion;
    private final int totalAppearancesCap;
    private final int minimalAppearanceIntervalMinutes;
    private final int availabilityDelayDays;
    private final int updateType;

    @SuppressWarnings("unused")
    public Configuration(
            @IntRange(from = 0) long targetVersion,
            @IntRange(from = 0) int totalAppearancesCap,
            @IntRange(from = 0) int minimalAppearanceIntervalMinutes,
            @IntRange int availabilityDelayDays,
            @AppUpdateType int updateType) throws ValidationException {
        this.targetVersion = targetVersion;
        this.totalAppearancesCap = totalAppearancesCap;
        this.minimalAppearanceIntervalMinutes = minimalAppearanceIntervalMinutes;
        this.availabilityDelayDays = availabilityDelayDays;
        this.updateType = updateType;
        validate();
    }

    public long getTargetVersion() {
        return targetVersion;
    }

    public int getTotalAppearancesCap() {
        return totalAppearancesCap;
    }

    public int getMinimalAppearanceIntervalMinutes() {
        return minimalAppearanceIntervalMinutes;
    }

    public int getAvailabilityDelayDays() {
        return availabilityDelayDays;
    }

    public int getUpdateType() {
        return updateType;
    }

    public void validate() throws ValidationException {
        Validators.isNonNegative("targetVersion", targetVersion);
        Validators.isNonNegative("totalAppearancesCap", totalAppearancesCap);
        Validators.isNonNegative("minimalAppearanceIntervalMinutes", minimalAppearanceIntervalMinutes);

        if (!Arrays.asList(AppUpdateType.FLEXIBLE, AppUpdateType.IMMEDIATE).contains(updateType)) {
            throw new ValidationException("invalid update type: " + updateType);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Configuration that = (Configuration) o;

        if (targetVersion != that.targetVersion) return false;
        if (totalAppearancesCap != that.totalAppearancesCap) return false;
        if (minimalAppearanceIntervalMinutes != that.minimalAppearanceIntervalMinutes) return false;
        if (availabilityDelayDays != that.availabilityDelayDays) return false;
        return updateType == that.updateType;
    }

    @Override
    public int hashCode() {
        int result = (int) (targetVersion ^ (targetVersion >>> 32));
        result = 31 * result + totalAppearancesCap;
        result = 31 * result + minimalAppearanceIntervalMinutes;
        result = 31 * result + availabilityDelayDays;
        result = 31 * result + updateType;
        return result;
    }

    @NonNull
    @Override
    public String toString() {
        return "{" +
                "targetVersion=" + targetVersion +
                ", totalAppearancesCap=" + totalAppearancesCap +
                ", minimalAppearanceIntervalMinutes=" + minimalAppearanceIntervalMinutes +
                ", availabilityDelayDays=" + availabilityDelayDays +
                ", updateType=" + updateType +
                '}';
    }
}
