/*
 * Copyright (c) 1997, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Distribution License v. 1.0, which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

package com.sun.xml.ws.api.addressing;

import com.sun.istack.NotNull;

import com.sun.xml.ws.api.WSBinding;
import com.sun.xml.ws.api.WSService;
import com.sun.xml.ws.api.message.Packet;
import com.sun.xml.ws.api.pipe.ClientTubeAssemblerContext;
import com.sun.xml.ws.api.pipe.Fiber;
import com.sun.xml.ws.api.pipe.TransportTubeFactory;
import com.sun.xml.ws.api.pipe.Tube;
import com.sun.xml.ws.api.server.WSEndpoint;
import com.sun.xml.ws.binding.BindingImpl;

/**
 * Delivers response messages targeted at non-anonymous endpoint addresses
 * @since 2.2.6
 */
public class NonAnonymousResponseProcessor {
	private static final NonAnonymousResponseProcessor DEFAULT = new NonAnonymousResponseProcessor();
	
	public static NonAnonymousResponseProcessor getDefault() {
		return DEFAULT;
	}
	
	protected NonAnonymousResponseProcessor() {}
	
    /**
     * Send a response to a non-anonymous address. Also closes the transport back channel
     * of {@link Packet} if it's not closed already.
     *
     * @param packet
     *      The response from our server, which will be delivered to the destination.
     * @return The response packet that should be used to complete the tubeline response processing
     */
	public Packet process(Packet packet) {
        Fiber.CompletionCallback fiberCallback = null;
        Fiber currentFiber = Fiber.getCurrentIfSet();
        if (currentFiber != null) {
            // Link completion of the current fiber to the new fiber that will
            // deliver the async response. This allows access to the response
            // packet that may be generated by sending a new message for the
            // current async response.

	        final Fiber.CompletionCallback currentFiberCallback =
	            currentFiber.getCompletionCallback();
	        
			if (currentFiberCallback != null) {
		          fiberCallback = new Fiber.CompletionCallback() {
		          public void onCompletion(@NotNull Packet response) {
		            currentFiberCallback.onCompletion(response);
		          }
		
		          public void onCompletion(@NotNull Throwable error) {
		            currentFiberCallback.onCompletion(error);
		          }
		        };
		        currentFiber.setCompletionCallback(null);
	        }
        }

        // we need to assemble a pipeline to talk to this endpoint.
		WSEndpoint<?> endpoint = packet.endpoint;
		WSBinding binding = endpoint.getBinding();
        Tube transport = TransportTubeFactory.create(Thread.currentThread().getContextClassLoader(),
            new ClientTubeAssemblerContext(
            		packet.endpointAddress, endpoint.getPort(), (WSService) null, 
            		binding, endpoint.getContainer(),
            		((BindingImpl) binding).createCodec(), null, null));
        Fiber fiber = endpoint.getEngine().createFiber();
        fiber.start(transport, packet, fiberCallback);
        
        // then we'll proceed the rest like one-way.
        Packet copy = packet.copy(false);
        copy.endpointAddress = null;
        
        return copy;
	}
}
