//    Copyright (c) 2014 - 2015 payu@india.com
//
//    Permission is hereby granted, free of charge, to any person obtaining a copy
//    of this software and associated documentation files (the "Software"), to deal
//    in the Software without restriction, including without limitation the rights
//    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//    copies of the Software, and to permit persons to whom the Software is
//    furnished to do so, subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be included in
//    all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//    THE SOFTWARE.

package com.payu.india.Tasks;

import android.os.AsyncTask;

import com.payu.india.Interfaces.CheckoutDetailsListener;
import com.payu.india.Model.Emi;
import com.payu.india.Model.PaymentDetails;
import com.payu.india.Model.PayuConfig;
import com.payu.india.Model.PayuResponse;
import com.payu.india.Model.PostData;
import com.payu.india.Model.Upi;
import com.payu.india.Payu.PayuConstants;
import com.payu.india.Payu.PayuErrors;
import com.payu.india.Payu.PayuUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.ProtocolException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;

import javax.net.ssl.HttpsURLConnection;


/**
 * Async task which takes care of Getting checkout details for a merchant
 * Takes PayuConfig as input and sends PayuResponse to the calling activity
 * Activity which calls {@link GetCheckoutDetailsTask} should implement {@link com.payu.india.Interfaces.CheckoutDetailsListener}
 */
public class GetCheckoutDetailsTask extends AsyncTask<PayuConfig, String, PayuResponse> {

    CheckoutDetailsListener mCheckoutDetailsListener;

    public GetCheckoutDetailsTask(CheckoutDetailsListener checkoutDetailsListener) {
        this.mCheckoutDetailsListener = checkoutDetailsListener;
    }

    @Override
    protected void onPreExecute() {
        super.onPreExecute();
    }

    @Override
    protected PayuResponse doInBackground(PayuConfig... params) {

        PayuResponse payuResponse = new PayuResponse();
        PostData postData = new PostData();

        try {
            URL url = null;
            // get the payuConfig first
            PayuConfig payuConfig = params[0];

            // set the environment
            switch (payuConfig.getEnvironment()) {
                case PayuConstants.PRODUCTION_ENV:
                    url = new URL(PayuConstants.PRODUCTION_FETCH_DATA_URL);
                    break;
                case PayuConstants.MOBILE_STAGING_ENV:
                    url = new URL(PayuConstants.MOBILE_TEST_FETCH_DATA_URL);
                    break;
                case PayuConstants.STAGING_ENV:
                    url = new URL(PayuConstants.TEST_FETCH_DATA_URL);
                    break;
                case PayuConstants.MOBILE_DEV_ENV:
                    url = new URL(PayuConstants.MOBILE_DEV_FETCH_DATA_URL);
                    break;
                case PayuConstants.BIZCHECKOUT_TEST_ENV:
                    url = new URL(PayuConstants.BIZ_CHECKOUT_TEST_FETCH_DATA_URL);
                    break;
                default:
                    url = new URL(PayuConstants.PRODUCTION_FETCH_DATA_URL);
                    break;
            }

            byte[] postParamsByte = payuConfig.getData().getBytes("UTF-8");

            HttpsURLConnection conn = PayuUtils.getHttpsConn(url.toString(), payuConfig.getData());
            if (null != conn) {

                InputStream responseInputStream = conn.getInputStream();

                StringBuffer responseStringBuffer = new StringBuffer();
                byte[] byteContainer = new byte[1024];
                for (int i; (i = responseInputStream.read(byteContainer)) != -1; ) {
                    responseStringBuffer.append(new String(byteContainer, 0, i));
                }

                JSONObject response = new JSONObject(responseStringBuffer.toString());

                if (response.has(PayuConstants.DETAILS) && response.optJSONObject(PayuConstants.DETAILS) != null) {

                    JSONObject paymentOptions = response.getJSONObject(PayuConstants.DETAILS).optJSONObject(PayuConstants.PAYMENT_OPTIONS);
                    if (paymentOptions != null && paymentOptions.length() > 0) {
                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_CC)) {
                            payuResponse.setCreditCard(prepareListWithKeyData(paymentOptions, PayuConstants.PAYU_CC));
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_DC)) {
                            payuResponse.setDebitCard(prepareListWithKeyData(paymentOptions, PayuConstants.PAYU_DC));
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_NB)) {
                            ArrayList<PaymentDetails> nbList = prepareListWithKeyData(paymentOptions, PayuConstants.PAYU_NB);
//                                Collections.sort(nbList, new PaymentDetailsComparator());
                            payuResponse.setNetBanks(nbList);
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.ENACH)) {
                            ArrayList<PaymentDetails> nbSiList = prepareListWithKeyData(paymentOptions, PayuConstants.ENACH);
                            payuResponse.setSiBankList(nbSiList);
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_SI)) {
                            ArrayList<PaymentDetails> siList = prepareListWithKeyData(paymentOptions, PayuConstants.PAYU_SI);
                            payuResponse.setStandingInstructions(siList);
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_CASH)) {
                            ArrayList<PaymentDetails> walletList = prepareListWithKeyData(paymentOptions, PayuConstants.PAYU_CASH);
                            PaymentDetails paymentDetails = getDetailsForKey(walletList, PayuConstants.PHONEPE_INTENT);
                            if (paymentDetails != null)
                                payuResponse.setPhonePe(paymentDetails);

                            Collections.sort(walletList, new PaymentDetailsComparator());
                            payuResponse.setCashCard(walletList);
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.IVR)) {
                            payuResponse.setIvr(prepareListWithKeyData(paymentOptions, PayuConstants.IVR));
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.IVRDC)) {
                            payuResponse.setIvrdc(prepareListWithKeyData(paymentOptions, PayuConstants.IVRDC));
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.UPI)) {
                            JSONObject upiObject = paymentOptions.getJSONObject(PayuConstants.UPI).getJSONObject(PayuConstants.PAYU_ALL);
                            if (upiObject.has((PayuConstants.UPI).toUpperCase())) {
                                JSONObject upiCollections = upiObject.getJSONObject((PayuConstants.UPI).toUpperCase());
                                Upi upi = new Upi();
                                upi.setTitle(upiCollections.optString(PayuConstants.TITLE));
                                upi.setAdditionalCharge(upiCollections.optString(PayuConstants.ADDITIONAL_CHARGE));
                                payuResponse.setUpi(upi);
                            }
                            if (upiObject.has((PayuConstants.TEZ).toUpperCase())) {
                                JSONObject upiCollections = upiObject.getJSONObject((PayuConstants.TEZ).toUpperCase());
                                Upi upi = new Upi();
                                upi.setTitle(upiCollections.optString(PayuConstants.TITLE));
                                upi.setAdditionalCharge(upiCollections.optString(PayuConstants.ADDITIONAL_CHARGE));
                                payuResponse.setGoogleTez(upi);
                            }
                            if (upiObject.has((PayuConstants.UPI_INTENT).toUpperCase())) {
                                JSONObject upiCollections = upiObject.getJSONObject((PayuConstants.UPI_INTENT).toUpperCase());
                                Upi upi = new Upi();
                                upi.setTitle(upiCollections.optString(PayuConstants.TITLE));
                                upi.setAdditionalCharge(upiCollections.optString(PayuConstants.ADDITIONAL_CHARGE));
                                payuResponse.setGenericIntent(upi);
                            }
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.EMI_IN_RESPONSE)) {
                            ArrayList<Emi> emiList = getEmiList(paymentOptions, PayuConstants.EMI_IN_RESPONSE);
                            payuResponse.setEmi(emiList);
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.LAZYPAY)) {
                            payuResponse.setLazyPay(prepareListWithKeyData(paymentOptions, PayuConstants.LAZYPAY));
                        }
                    }
                }

                if (response.has(PayuConstants.STATUS) && response.getString(PayuConstants.STATUS).contentEquals("0")) {
                    postData = new PostData();
                    postData.setCode(PayuErrors.INVALID_HASH);
                    postData.setStatus(PayuConstants.ERROR);
                    postData.setResult(response.getString(PayuConstants.MSG));
                } else {
                    postData.setCode(PayuErrors.NO_ERROR);
                    // Result status should be success and the status of user cards
                    postData.setResult(PayuErrors.SDK_DETAILS_FETCHED_SUCCESSFULLY);
                    if (response.has(PayuConstants.USERCARDS) && response.getJSONObject(PayuConstants.USERCARDS).has(PayuConstants.MSG)) {
                        postData.setResult(postData.getResult() + " " + response.getJSONObject(PayuConstants.USERCARDS).get(PayuConstants.MSG).toString());
                    }
                    postData.setStatus(PayuConstants.SUCCESS);
                }
            }
        } catch (ProtocolException e) {
            postData.setCode(PayuErrors.PROTOCOL_EXCEPTION);
            postData.setStatus(PayuConstants.ERROR);
            postData.setResult(e.getMessage());
        } catch (UnsupportedEncodingException e) {
            postData.setCode(PayuErrors.UN_SUPPORTED_ENCODING_EXCEPTION);
            postData.setStatus(PayuConstants.ERROR);
            postData.setResult(e.getMessage());
        } catch (JSONException e) {
            postData.setCode(PayuErrors.JSON_EXCEPTION);
            postData.setStatus(PayuConstants.ERROR);
            postData.setResult(e.getMessage());
        } catch (IOException e) {
            postData.setCode(PayuErrors.IO_EXCEPTION);
            postData.setStatus(PayuConstants.ERROR);
            postData.setResult(e.getMessage());
        }

        payuResponse.setResponseStatus(postData);
        return payuResponse;
    }

    private ArrayList<Emi> getEmiList(JSONObject paymentOptions, String key) throws JSONException {
        JSONObject ccObject = paymentOptions.getJSONObject(key).getJSONObject(PayuConstants.PAYU_ALL).optJSONObject(PayuConstants.PAYU_CC);
        if (ccObject != null && ccObject.optJSONObject(PayuConstants.PAYU_ALL) != null) {
            JSONObject allObject = ccObject.getJSONObject(PayuConstants.PAYU_ALL);
            Iterator<String> keysIterator = allObject.keys();
            ArrayList<Emi> emiList = new ArrayList<Emi>();
            while (keysIterator.hasNext()) {
                String code = keysIterator.next();
                JSONObject emiObject = allObject.getJSONObject(code);
                JSONObject tenureOptions = emiObject.getJSONObject(PayuConstants.TENURE_OPTIONS);
                Iterator<String> tenureKeysIterator = tenureOptions.keys();
                while (tenureKeysIterator.hasNext()) {
                    String bankCode = tenureKeysIterator.next();
                    JSONObject tenureObject = tenureOptions.getJSONObject(bankCode);
                    Emi emi = new Emi();
                    emi.setBankCode(bankCode);
                    emi.setAdditionalCharge(tenureObject.optString(PayuConstants.ADDITIONAL_CHARGE));
                    emiList.add(emi);
                }
            }
//                Collections.sort(emiList, new EmiComparator());
            return emiList;
        }
        return null;
    }

    @Override
    protected void onPostExecute(PayuResponse payuResponse) {
        super.onPostExecute(payuResponse);
        mCheckoutDetailsListener.onCheckoutDetailsDetailsResponse(payuResponse);
    }

    class EmiComparator implements Comparator<Emi> {

        @Override
        public int compare(Emi p1, Emi p2) {

            return p1.getBankName().compareTo(p2.getBankName());

        }
    }

    class PaymentDetailsComparator implements Comparator<PaymentDetails> {

        @Override
        public int compare(PaymentDetails p1, PaymentDetails p2) {

            return p1.getBankName().compareTo(p2.getBankName());

        }
    }

    private PaymentDetails getDetailsForKey(ArrayList<PaymentDetails> paymentDetailsList, String key) {
        if (paymentDetailsList == null || paymentDetailsList.size() == 0 || key == null || key.isEmpty())
            return null;

        for (PaymentDetails paymentDetails : paymentDetailsList) {
            if (paymentDetails.getBankCode().equalsIgnoreCase(key))
                return paymentDetails;
        }
        return null;
    }

    private ArrayList<PaymentDetails> prepareListWithKeyData(JSONObject paymentOptions, String key) throws JSONException {
        JSONObject allObject = paymentOptions.getJSONObject(key).getJSONObject(PayuConstants.PAYU_ALL);
        ArrayList<PaymentDetails> paymentDetailsList = new ArrayList();
        Iterator<String> keysIterator = allObject.keys();
        while (keysIterator.hasNext()) {
            String bankCode = keysIterator.next();
            JSONObject ccObject = allObject.getJSONObject(bankCode);
            PaymentDetails paymentDetails = new PaymentDetails();
            paymentDetails.setBankCode(bankCode);
            paymentDetails.setBankName(ccObject.optString(PayuConstants.TITLE));
            paymentDetails.setAdditionalCharge(ccObject.optString(PayuConstants.ADDITIONAL_CHARGE));
            paymentDetailsList.add(paymentDetails);
        }
        return paymentDetailsList;
    }

    private boolean isJSONObjectAvailableForKey(JSONObject paymentOptions, String key) {
        return paymentOptions.has(key) && paymentOptions.optJSONObject(key) != null && paymentOptions.optJSONObject(key).optJSONObject(PayuConstants.PAYU_ALL) != null;
    }
}
