package com.payu.otpparser

import android.Manifest
import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import com.google.android.gms.auth.api.phone.SmsRetriever
import com.payu.payuanalytics.analytics.PayUAnalytics

class OtpHandler constructor(var activity: ComponentActivity, var otpHandlerCallback: OtpHandlerCallback) : DefaultLifecycleObserver,
    SMSReceiver.SmsBroadcastReceiverListener {
    private var smsBroadcastReceiver: SMSReceiver? = null
    private val REQUEST_CODE = 1003
    private var payUAnalytics: PayUAnalytics? = null
    private var merchantKey = ""
    private var txnId = ""
    fun setBundle(bundle: Bundle) {
        if (null != bundle[Constants.MERCHANT_KEY])
            merchantKey = bundle[Constants.MERCHANT_KEY] as String
        if (null != bundle[Constants.TXNID])
            txnId = bundle[Constants.TXNID] as String

    }
    fun startListening() {
        payUAnalytics =
            PayUAnalytics.getInstance(activity, "otp_parser_local_cache_analytics")

        if (!Utils.isPermissionAvailable(activity)) {
            if (!Utils.checkSmsPermissionAvailable(activity)) {
                L.v("startSmsUserConsent")
                addAnalytics(
                    Constants.OTP_PARSER_SDK_EVENT,
                    "requested_consent_permission"
                )
                startSmsUserConsent()
            } else {

                if ( Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                        if ( Utils.getPermissionDenyingStatusSharedPref(activity, Manifest.permission.RECEIVE_SMS) &&! Utils.checkNeverAskAgainSelectedRationale(activity, Manifest.permission.RECEIVE_SMS)){
                            startSmsUserConsent()
                            addAnalytics(
                                Constants.OTP_PARSER_SDK_EVENT,
                                "requested_consent_permission"
                            )
                            L.v("startSmsUserConsent after dont ask again")
                        }
                    else{
                            addAnalytics(Constants.OTP_PARSER_SDK_EVENT,
                                "requested_sms_permission")
                            L.v("requestPermissions")
                            ActivityCompat.requestPermissions(
                                activity,
                                arrayOf(Manifest.permission.RECEIVE_SMS), REQUEST_CODE)
                        }
                    }


            }
        } else {
            L.v("registerSMSReceiver")
            addAnalytics(Constants.OTP_PARSER_SDK_EVENT,
                "permission_granted_already")
            registerSMSReceiver()
        }
    }

    private fun startSmsUserConsent() {
        SmsRetriever.getClient(activity).also {
            //We can add user phone number or leave it blank
            it.startSmsUserConsent(null)
                .addOnSuccessListener {
                    L.v( "SmsRetriever success")
                    registerConsentReceiver()
                }
                .addOnFailureListener {
                    L.v( "SmsRetriever failure" + it.message)

                    //TODO Log analytics here
                }
        }
    }

    fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        L.v( "onActivityResult")
        if(resultCode == Activity.RESULT_OK && null!=data) {
            val message = data?.getStringExtra(SmsRetriever.EXTRA_SMS_MESSAGE)
            val code = message?.let { Utils.fetchVerificationCode(it) }
            addAnalytics(Constants.OTP_PARSER_SDK_EVENT,
                "otp_fetched_through_consent")
            PayUOtpParserConfig.otpCallback?.onOtpReceived(code!!)
        }else if(resultCode == Activity.RESULT_CANCELED){
            addAnalytics(Constants.OTP_PARSER_SDK_EVENT,
                "user_denied_through_consent")
            PayUOtpParserConfig.otpCallback?.onUserDenied()

        }
        unregisterReceiver()
    }

    fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        when (requestCode) {
            REQUEST_CODE -> {
                if (grantResults.isNotEmpty() && grantResults[0] ==
                    PackageManager.PERMISSION_GRANTED
                ) {
                    if ((ContextCompat.checkSelfPermission(
                            activity,
                            Manifest.permission.RECEIVE_SMS
                        ) ===
                                PackageManager.PERMISSION_GRANTED)
                    ) {
                        L.v("Runtime Permission Granted")
                        addAnalytics(Constants.OTP_PARSER_SDK_EVENT,
                            "permission_granted_sms_permission")
                        registerSMSReceiver()
                    }
                } else {
                    L.v("Runtime Permission Denyied ")
                    addAnalytics(Constants.OTP_PARSER_SDK_EVENT,
                        "permission_denied_sms_permission")
                    if (null!= activity && !activity?.isFinishing)
                    Utils.setPermissionDenyingStatus(activity, Manifest.permission.RECEIVE_SMS)
                    PayUOtpParserConfig.otpCallback?.onUserDenied()
                }

            }
        }

    }

    fun registerSMSReceiver() {
        if (null == smsBroadcastReceiver) {
            smsBroadcastReceiver = SMSReceiver(this)
            val intentFilter = IntentFilter("android.provider.Telephony.SMS_RECEIVED")
            intentFilter.priority = 999
            activity.registerReceiver(smsBroadcastReceiver,intentFilter)
        }
    }

    fun registerConsentReceiver() {
        if (null == smsBroadcastReceiver) {
//            L.v("registerReceiver")
            smsBroadcastReceiver = SMSReceiver(this)
            val intentFilter = IntentFilter(SmsRetriever.SMS_RETRIEVED_ACTION)
            activity.registerReceiver(
                smsBroadcastReceiver,
                intentFilter,
                SmsRetriever.SEND_PERMISSION, null
            )
        }
    }

    override fun onSuccess(intent: Intent?) {
        L.v( "onsuccess")
        try {
            activity.startActivityForResult(intent, 101)
        } catch (e: ActivityNotFoundException) {
            e.printStackTrace()
        }
    }

    override fun onSuccess(message: String?) {
        L.v("onSuccess sms permission")
        val code = Utils.fetchVerificationCode(message!!)
        addAnalytics(Constants.OTP_PARSER_SDK_EVENT,
            "otp_fetched_through_receiver")
        PayUOtpParserConfig.otpCallback?.onOtpReceived(code)
        unregisterReceiver()
    }

    override fun onFailure() {
        L.v( "onFailure")
        PayUOtpParserConfig.otpCallback?.onUserDenied()
        unregisterReceiver()
    }

    override fun onDestroy(owner: LifecycleOwner) {
        L.v( "onDestroy")
        unregisterReceiver()
        activity.lifecycle.removeObserver(this)
        otpHandlerCallback.lifeCycleOnDestroy()
    }

    private fun unregisterReceiver() {
        if (null != smsBroadcastReceiver) {
            activity.unregisterReceiver(smsBroadcastReceiver)
            smsBroadcastReceiver = null
            L.v("unregisterReceiver")
        }
    }

    private fun addAnalytics(key:String, value:String){
        payUAnalytics?.log(
            Utils.getLogMessage(
                activity,
                key,
                value,
                merchantKey,
                txnId
            )
        )
    }
}