// The MIT License (MIT)
//
// Copyright (c) 2014-2015 PayU India
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.payu.custombrowser;

import android.annotation.TargetApi;
import android.graphics.Bitmap;
import android.net.http.SslError;
import android.os.Build;
import androidx.annotation.NonNull;
import android.text.TextUtils;
import android.webkit.SslErrorHandler;
import android.webkit.WebResourceError;
import android.webkit.WebResourceRequest;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.payu.custombrowser.bean.CustomBrowserData;
import com.payu.custombrowser.util.CBConstant;
import com.payu.custombrowser.util.CBUtil;


/**
 * Created by Shoaib on 2016/06/16.
 * Custom PayU snooze web view client.
 */
public class PayUSurePayWebViewClient extends WebViewClient {

    private boolean loadingFinished = true;
    private boolean redirect = false;
    private String mainUrl = "";
    private Bank bank;

    /**
     * Customize payu web view controller for merchants.
     *
     * @param bank name of the  bank
     */
    public PayUSurePayWebViewClient(@NonNull Bank bank, @NonNull String merchantKey) {
        this.bank = bank;
        if (Bank.keyAnalytics == null)
            Bank.keyAnalytics = merchantKey;

    }

    @Override
    public void onPageStarted(WebView view, String url, Bitmap favicon) {
        super.onPageStarted(view, url, favicon);
        loadingFinished = false;
        if (bank != null) {
            bank.onPageStartedWebclient(url); // set the current url
        }
    }

    @Override
    public void onPageFinished(WebView view, String url) {
        super.onPageFinished(view, url);
if (null!=CBUtil.getDomain(url))
            CustomBrowserData.SINGLETON.addDomain(CBUtil.getDomain(url));


        if (!redirect) {
            loadingFinished = true;
        }

        if (url.equals(mainUrl)) {
            loadingFinished = true;
            redirect = false;
        } else {
            redirect = false;
        }
        if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT && !bank.isSurePayValueLoaded && bank.isS2SHtmlSupport) {
            String js = new CBUtil().getStringSharedPreference(bank.getContext(),CBConstant.SUREPAY_S2S);
            if(!TextUtils.isEmpty(js)){
                view.loadUrl("javascript:"+js+"()");
            }else {
                view.evaluateJavascript(bank.getResources().getString(R.string.surepay_js), null);
            }
        }
        if (bank != null) {
            bank.onPageFinishWebclient(url); // notify bank.java that page loading is done.
        }
    }

    @Override
    public void onLoadResource(WebView view, String url) {
        if (bank != null)
            bank.onLoadResourse(view, url);
        super.onLoadResource(view, url);

    }


    @Override
    public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
        super.onReceivedError(view, errorCode, description, failingUrl);
        if (failingUrl != null) {
            // Show sure pay only if the error description is not ERROR_CONNECT.
            // Amex pages randomly returns error connect , dont show sure pay.
            if (bank != null && !description.contentEquals(CBConstant.ERR_CONNECTION_RESET) && !description.contentEquals(CBConstant.ERR_NAME_NOT_RESOLVED )) {
                if (Build.VERSION.SDK_INT < 23) {
                    bank.onReceivedErrorWebClient(errorCode, description);
                }
            }
        }
    }

    @TargetApi(23)
    @Override
    public void onReceivedError(WebView view, WebResourceRequest request, WebResourceError error) {
        super.onReceivedError(view, request, error);
        // TODO - we have to move CBConstant.ERR_CONNECTION_RESET) to cbutils.
        if (bank != null && !error.getDescription().toString().contentEquals(CBConstant.ERR_CONNECTION_RESET) && !error.getDescription().toString().contentEquals(CBConstant.ERR_NAME_NOT_RESOLVED)) { // Amex page returns ERROR_CONNECT
            bank.onReceivedErrorWebClient(error.getErrorCode(), error.getDescription().toString());
        }
    }


    // TODO : Remove the inner if else
    @Override
    public void onReceivedSslError(WebView view, SslErrorHandler handler, SslError error) {
        bank.onReceivedSslError(view, handler, error);
//        super.onReceivedSslError(view, handler, error);
//        if (Build.VERSION.SDK_INT <= 10) {
//            handler.proceed();
//        }
    }

    @Override
    public boolean shouldOverrideUrlLoading(WebView view, String url) {
        this.mainUrl = url;

        if(CBUtil.isPlayStoreUrl(url)){
            CBUtil.launchPlayStore(bank.getContext(), url, CBUtil.getWebViewVersion(view));
            // returning true to tell the webview not to load this url,
            // since we have taken care of it.
            return true;
        }
        if(url.startsWith(CBConstant.DEEP_LINK_INTENT_URI)) {
            //In case of webview don't entertain deep link intent
            return true;
        }
        if (!loadingFinished) {
            redirect = true;
        }

        loadingFinished = false;

        if (bank != null) {
            bank.onOverrideURL(url);
        }

        return false;
    }

}