// The MIT License (MIT)
//
// Copyright (c) 2014-2015 PayU India
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package com.payu.custombrowser;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.TrafficStats;
import android.net.wifi.ScanResult;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.telephony.CellInfo;
import android.telephony.CellInfoCdma;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellInfoWcdma;
import android.telephony.CellSignalStrengthCdma;
import android.telephony.CellSignalStrengthGsm;
import android.telephony.CellSignalStrengthLte;
import android.telephony.CellSignalStrengthWcdma;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.animation.AlphaAnimation;

import org.json.JSONObject;

import java.util.List;
import java.util.StringTokenizer;

/**
 * Created by minie on 29/6/15.
 * Contains cb's utility functions.
 */
public class CBUtil {

    public static  boolean getbooleanSharedPreference(String key,Activity mActivity) {
        SharedPreferences sharedPreferences = mActivity.getSharedPreferences("com.payu", Context.MODE_PRIVATE);
        return sharedPreferences.getBoolean(key, false);
    }

    public static void setBooleanSharedPreference(String key, boolean value,Activity mActivity ) {
        SharedPreferences.Editor editor = mActivity.getSharedPreferences("com.payu", Context.MODE_PRIVATE).edit();
        editor.putBoolean(key, value);
        editor.commit();
    }

    /**
     * Function to find whether network is available or not.
     * @param context
     * @return
     */
    static boolean isNetworkAvailable(Context context) {
        boolean haveConnectedWifi = false;
        boolean haveConnectedMobile = false;
        ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo[] netInfo = cm.getAllNetworkInfo();
        for (NetworkInfo ni : netInfo) {
            if (ni.getTypeName().equalsIgnoreCase("WIFI"))
                if (ni.isConnected())
                    haveConnectedWifi = true;
            if (ni.getTypeName().equalsIgnoreCase("MOBILE"))
                if (ni.isConnected())
                    haveConnectedMobile = true;
        }
        return haveConnectedWifi || haveConnectedMobile;
    }

    static String getDeviceDensity(Activity activity){
        DisplayMetrics metrics = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(metrics);
        return metrics.densityDpi+"";
    }

    static void getDownloadSpeed(Activity activity){
        String testing[]=new String[2];
        long BeforeTime = System.currentTimeMillis();
        long TotalTxBeforeTest = TrafficStats.getTotalTxBytes();
        long TotalRxBeforeTest = TrafficStats.getTotalRxBytes();

        long TotalTxAfterTest = TrafficStats.getTotalTxBytes();
        long TotalRxAfterTest = TrafficStats.getTotalRxBytes();
        long AfterTime = System.currentTimeMillis();

        double TimeDifference = AfterTime - BeforeTime;

        double rxDiff = TotalRxAfterTest - TotalRxBeforeTest;
        double txDiff = TotalTxAfterTest - TotalTxBeforeTest;

        if((rxDiff != 0) && (txDiff != 0)) {
            double rxBPS = (rxDiff / (TimeDifference/1000)); // total rx bytes per second.
            double txBPS = (txDiff / (TimeDifference/1000)); // total tx bytes per second.
            testing[0] = String.valueOf(rxBPS) + "bps. Total rx = " + rxDiff;
            testing[1] = String.valueOf(txBPS) + "bps. Total tx = " + txDiff;
        }
        else {
            testing[0] = "No uploaded or downloaded bytes.";
        }
    }

    static  void setAlpha(float alpha,View view){
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.HONEYCOMB) {
            final AlphaAnimation animation = new AlphaAnimation(alpha, alpha);
            animation.setDuration(10);
            animation.setFillAfter(true);
            view.startAnimation(animation);
        } else {
            view.setAlpha(alpha);
        }
    }

    static String getNetworkStatus(Activity activity) {
        try {
            if (null != activity && !activity.isFinishing()) {
                ConnectivityManager cm = (ConnectivityManager) activity.getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkInfo info = cm.getActiveNetworkInfo();
                if (info == null || !info.isConnected())
                    return "Not connected"; //not connected
                if (info.getType() == ConnectivityManager.TYPE_WIFI)
                    return "WIFI";
                if (info.getType() == ConnectivityManager.TYPE_MOBILE) {
                    int networkType = info.getSubtype();
                    switch (networkType) {
                        case TelephonyManager.NETWORK_TYPE_GPRS:
                            return "GPRS";
                        case TelephonyManager.NETWORK_TYPE_EDGE:
                            return "EDGE";
                        case TelephonyManager.NETWORK_TYPE_CDMA:
                            return "CDMA";
                        case TelephonyManager.NETWORK_TYPE_1xRTT:
                        case TelephonyManager.NETWORK_TYPE_IDEN: //api<8 : replace by 11
                            return "2G";
                        case TelephonyManager.NETWORK_TYPE_UMTS:
                        case TelephonyManager.NETWORK_TYPE_EVDO_0:
                        case TelephonyManager.NETWORK_TYPE_EVDO_A:
                        case TelephonyManager.NETWORK_TYPE_HSDPA:
                        case TelephonyManager.NETWORK_TYPE_HSUPA:
                        case TelephonyManager.NETWORK_TYPE_HSPA:
                            return "HSPA";
                        case TelephonyManager.NETWORK_TYPE_EVDO_B: //api<9 : replace by 14
                        case TelephonyManager.NETWORK_TYPE_EHRPD:  //api<11 : replace by 12
                        case TelephonyManager.NETWORK_TYPE_HSPAP:  //api<13 : replace by 15
                            return "3G";
                        case TelephonyManager.NETWORK_TYPE_LTE:    //api<11 : replace by 13
                            return "4G";
                        default:
                            return "?";
                    }
                }
            }
        }catch (Exception e) {
            return "?";
        }
        return "?";
    }

    static  int getNetworkStrenght(Activity activity){
        try {
            ConnectivityManager connectivityManager = (ConnectivityManager) activity.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo[] netInfo = connectivityManager.getAllNetworkInfo();
            TelephonyManager telephonyManager = (TelephonyManager) activity.getSystemService(Context.TELEPHONY_SERVICE);

            int strength = 0;

            for (NetworkInfo networkInfo : netInfo) {
                // we are interested only in mobile.
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2 && networkInfo.getTypeName().equalsIgnoreCase("MOBILE") && networkInfo.isConnected()) { // okay now its mobile network
                    // lets find the strength.
                    for (CellInfo info : telephonyManager.getAllCellInfo()) {
                        if (info.isRegistered()) { // connected
                            if (info instanceof CellInfoGsm) {
                                CellSignalStrengthGsm gsm = ((CellInfoGsm) info).getCellSignalStrength();
                                strength = gsm.getDbm();
                            } else if (info instanceof CellInfoCdma) {
                                CellSignalStrengthCdma cdma = ((CellInfoCdma) info).getCellSignalStrength();
                                strength = cdma.getDbm();
                            } else if (info instanceof CellInfoLte) {
                                CellSignalStrengthLte lte = ((CellInfoLte) info).getCellSignalStrength();
                                strength = lte.getDbm();
                            } else if (info instanceof CellInfoWcdma) {
                                CellSignalStrengthWcdma wcdma = ((CellInfoWcdma) info).getCellSignalStrength(); // jelly bean mr2
                                strength = wcdma.getDbm();
                            }
                        }
                    }
                }
            }
            return strength;
        }catch (Exception e){
            return 0;
        }

    }

    static  void setStringSharedPreference(Activity activity,String key,String url){
        String str=getStringSharedPreference(activity, key);
        if(str.equalsIgnoreCase("")){
            str=url;
        }else if(str.indexOf("||")==-1){
            str=str+"||"+url;
        }else{
            StringTokenizer st = new StringTokenizer(str,"||");
            st.nextToken();
            str=st.nextToken()+"||"+url;
        }
        SharedPreferences.Editor sharedPreferencesEditor = (activity.getSharedPreferences("com.payu.custombrowser",activity.MODE_PRIVATE)).edit();
        sharedPreferencesEditor.putString(key,str);
        sharedPreferencesEditor.commit();
    }

    static  String getStringSharedPreference(Activity activity,String key){
        SharedPreferences sharedPreferences= (activity.getSharedPreferences("com.payu.custombrowser",activity.MODE_PRIVATE));
        return sharedPreferences.getString(key,"");
    }
    static void delSharedLastURL(Activity activity,String key){
        try {
            SharedPreferences.Editor sharedPreferencesEditor = (activity.getSharedPreferences("com.payu.custombrowser", activity.MODE_PRIVATE)).edit();
            sharedPreferencesEditor.remove(key);
            sharedPreferencesEditor.commit();
        }catch(Exception e)
        {
            e.printStackTrace();
        }
    }


    static String updateLastUrl(String lastUrl)
    {
        try {
            if (!(lastUrl.contains("||"))) {
                if (lastUrl.length() > 128)
                    return lastUrl.substring(0, 127);
            } else {
                StringTokenizer st = new StringTokenizer(lastUrl, "||");
                String firstURl = st.nextToken();
                String secondUrl = st.nextToken();
                if (firstURl.length() > 128)
                    firstURl = firstURl.substring(0, 125);
                if (secondUrl.length() > 128)
                    secondUrl = secondUrl.substring(0, 125);
                return firstURl + "||" + secondUrl;
            }

        }catch(Exception e)
        {
            e.printStackTrace();
            return "";

        }
        return "";
    }

    static void storeInSharedPreferences(Context context, String key, String value){
        SharedPreferences sharedPreferences = context.getSharedPreferences(CBConstant.PAYU_PREFERENCE, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putString(key, value);
        editor.commit();
    }

    public String getFromSharedPreferences(Context context, String key){
        return getFromSharedPreferences(context, key, CBConstant.DEFAULT);
    }

    public String getFromSharedPreferences(Context context, String key, String defaultValue){
        SharedPreferences sharedPreferences = context.getSharedPreferences(CBConstant.PAYU_PREFERENCE, Context.MODE_PRIVATE);
        return sharedPreferences.getString(key, defaultValue);
    }

    public void removeFromSharedPreferences(Context context, String key){
        SharedPreferences sharedPreferences = context.getSharedPreferences(CBConstant.PAYU_PREFERENCE, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.remove(key);
        editor.apply();
    }

}
