package com.ontotext.trree;

import com.ontotext.license.License;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

/**
 * Class that's used by maven resource filtering to generate proper metadata information about the current build.
 * Note that this copies some of the information from release.properties but allows eclipse developers to not recompile
 * just to get the proper setup.
 */
public class ReleaseInfo {
	private static final String TIMESTAMP_FORMAT = "yyyyMMddHHmmss";

	private static final String PROPERTY_VERSION_VALUE = "${parsedVersion.majorVersion}.${parsedVersion.minorVersion}";
	private static final String PROPERTY_REVISION_VALUE = "${buildNumber}";
	private static final String PROPERTY_FULL_VERSION_VALUE = "${version}";
	private static final String PROPERTY_TIMESTAMP_VALUE = "${timestamp}";

    private static final Logger LOGGER = LoggerFactory.getLogger(ReleaseInfo.class);

    private String version;
	private final int versionNumber;
	private final int revision;
	private final Date timestamp;
	private final String rawRevision;
	private String fullVersion;
	private static ReleaseInfo instance = new ReleaseInfo();
	/// singleton
	public static ReleaseInfo get() {
		return instance;
	}

	private ReleaseInfo() {

		final Properties releaseProps = new Properties();
		final InputStream releasePropsStream = Thread.currentThread().getContextClassLoader()
				.getResourceAsStream("META-INF/release.properties");
		if (releasePropsStream != null) {
			try {
				releaseProps.load(releasePropsStream);
			} catch (IOException e) {
				LOGGER.warn("Couldn't find release properties, defaults will be used: " + e.getMessage());
			} finally {
				IOUtils.closeQuietly(releasePropsStream);
			}
		}

		version = PROPERTY_VERSION_VALUE;
		if (version.isEmpty() || version.startsWith("$")) {
			version = "10.8";
		}
		fullVersion = PROPERTY_FULL_VERSION_VALUE;
		if (fullVersion.isEmpty() || fullVersion.startsWith("$")) {
			fullVersion = "10.8-SNAPSHOT";
		}
		String[] versionItems = version.split("\\.", 2);
		versionNumber = Integer.parseInt(versionItems[0]) * 100 + Integer.parseInt(versionItems[1]);
		rawRevision = PROPERTY_REVISION_VALUE;
		revision = parseRevision(PROPERTY_REVISION_VALUE);
        Date timestamp = new Date();
		try {
			timestamp = new SimpleDateFormat(TIMESTAMP_FORMAT).parse(PROPERTY_TIMESTAMP_VALUE);
		} catch (ParseException e) {
			LOGGER.warn("Couldn't parse timestamp " + PROPERTY_TIMESTAMP_VALUE + " properly!");
		}
		this.timestamp = timestamp;
	}

    private static int parseRevision(String prop) {
        try {
        if (prop.startsWith("0x")) {
            // Revisions larger than 7fffffff can't be parsed as integers, that's why we
            // parse them as longs and cast them to ints. This will may produce negative
            // revision numbers.
            return (int) Long.parseLong(prop.substring(2), 16);
        } else {
            return Integer.parseInt(prop);
        }
        } catch (NumberFormatException e) {
            LOGGER.error("Bad revision format: ", prop, e);
            return 0;
        }
    }

	/**
	 * Get a full version of the database. This is the version of the database + the build number.
	 * See http://semver.org/ (section 10) for more details on the format.
	 * @return the
	 */
	public String getFullVersion() {
		// the version has build metadata
		if (fullVersion.contains("+")) {
			return fullVersion;
		}

		return fullVersion + "+sha." + getRawRevision();
	}

	public String getRawRevision() {
		if (rawRevision.startsWith("0x")) {
			return rawRevision.substring(2);
		}
		return rawRevision;
	}

	public String getVersion() {
		return version;
	}

	public int getVersionNumber() { return versionNumber; };

	public int getRevision() {
		return revision;
	}

	public Date getTimestamp() {
		return timestamp;
	}
}
