/*
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.olekdia.datetimepickers;

import android.animation.Keyframe;
import android.animation.ObjectAnimator;
import android.animation.PropertyValuesHolder;
import android.content.Context;
import android.os.Build;
import android.util.TypedValue;
import android.view.View;

import com.olekdia.androidcommon.extensions.ResourcesExtensionsKt;

/**
 * Utility helper functions for time and date pickers.
 */
public class DTPickersHelper {

    public static final int DAYS_IN_WEEK = 7;
    public static final int MONTHS_IN_YEAR = 12;

    public static final int AM = 0;
    public static final int PM = 1;


    public static final int PULSE_ANIMATOR_DURATION = 544;

    /**
     * Render an animator to pulsate a view in place.
     *
     * @param labelToAnimate the view to pulsate.
     * @return The animator object. Use .start() to begin.
     */
    public static ObjectAnimator getPulseAnimator(final View labelToAnimate,
                                                  final float decreaseRatio,
                                                  final float increaseRatio) {
        final Keyframe k0 = Keyframe.ofFloat(0f, 1f);
        final Keyframe k1 = Keyframe.ofFloat(0.275f, decreaseRatio);
        final Keyframe k2 = Keyframe.ofFloat(0.69f, increaseRatio);
        final Keyframe k3 = Keyframe.ofFloat(1f, 1f);

        final PropertyValuesHolder scaleX = PropertyValuesHolder.ofKeyframe("scaleX", k0, k1, k2, k3);
        final PropertyValuesHolder scaleY = PropertyValuesHolder.ofKeyframe("scaleY", k0, k1, k2, k3);
        final ObjectAnimator pulseAnimator = ObjectAnimator.ofPropertyValuesHolder(labelToAnimate, scaleX, scaleY);
        pulseAnimator.setDuration(PULSE_ANIMATOR_DURATION);

        return pulseAnimator;
    }

    /**
     * Gets the colorAccent from the current context, if possible/available
     *
     * @param context The context to use as reference for the color
     * @return the accent color of the current context
     */
    public static int getAccentColorFromThemeIfAvailable(final Context context) {
        TypedValue typedValue = new TypedValue();
        // First, try the android:colorAccent
        if (Build.VERSION.SDK_INT >= 21) {
            context.getTheme().resolveAttribute(android.R.attr.colorAccent, typedValue, true);
            return typedValue.data;
        }
        // Next, try colorAccent from support lib
        int colorAccentResId = context.getResources().getIdentifier("colorAccent", "attr", context.getPackageName());
        if (colorAccentResId != 0 && context.getTheme().resolveAttribute(colorAccentResId, typedValue, true)) {
            return typedValue.data;
        }
        // Return the value in mdtp_accent_color
        return ResourcesExtensionsKt.getColorCompat(context, R.color.mdtp_accent_color);
    }

    public static int getPrimaryColorFromThemeIfAvailable(final Context context) {
        TypedValue typedValue = new TypedValue();
        // First, try the android:colorAccent
        if (Build.VERSION.SDK_INT >= 21) {
            context.getTheme().resolveAttribute(android.R.attr.colorPrimary, typedValue, true);
            return typedValue.data;
        }
        // Next, try colorAccent from support lib
        int colorAccentResId = context.getResources().getIdentifier("colorPrimary", "attr", context.getPackageName());
        if (colorAccentResId != 0 && context.getTheme().resolveAttribute(colorAccentResId, typedValue, true)) {
            return typedValue.data;
        }
        // Return the value in mdtp_accent_color
        return ResourcesExtensionsKt.getColorCompat(context, R.color.mdtp_primary_color);
    }

    /**
     * Gets dialog type (Light/Dark) from current theme
     *
     * @param context The context to use as reference for the boolean
     * @param current Default value to return if cannot resolve the attribute
     * @return true if dark mode, false if light.
     */
    public static boolean isDarkTheme(final Context context, final boolean current) {
        return ResourcesExtensionsKt.resolveBoolean(context, R.attr.mdtp_theme_dark, current);
    }

    public static int indexDayToShiftedWeekIndexDay(final int indexOfCalDay, final int weekStart) {
        return (((indexOfCalDay + weekStart) % DAYS_IN_WEEK) + 5) % DAYS_IN_WEEK;
    }
}
