package com.olekdia.commonhelpers;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.content.res.TypedArray;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.support.annotation.AttrRes;
import android.support.annotation.IntDef;
import android.support.annotation.NonNull;
import android.support.annotation.StyleRes;
import android.support.v4.util.SimpleArrayMap;
import android.text.Html;
import android.text.format.DateFormat;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.View;
import android.widget.TextView;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Formatter;
import java.util.Locale;

public class CommonHelper {

    public static final int RESOURCE_NOT_FOUND = 0;
    public static final int INVALID = -1;

    public static final char ZERO_DIGIT = '0';
    public static final char NINE_DIGIT = '9';

    public static final int DAYS_IN_WEEK = 7;
    public static final int MONTHS_IN_YEAR = 12;

    /**
     * Unicode "Left-To-Right Embedding"
     */
    public static final char LRE = '\u202A';
    /**
     * Unicode "Right-To-Left Embedding"
     */
    public static final char RLE = '\u202B';
    /**
     * Unicode "Pop Directional Formatting"
     */
    public static final char PDF = '\u202C';
    /**
     * Unicode "Pop Directional Isolate"
     */
    public static final char PDI = '\u2069';
    /**
     * Left-To-Right Mark
     */
    public static final char LRM = '\u200E';
    /**
     * Right-To-Left Mark
     */
    public static final char RLM = '\u200F';
    /**
     * Left-To-Right Isolated
     */
    public static final char LRI = '\u2066';
    /**
     * Right-To-Left Isolated
     */
    public static final char RLI = '\u2067';
    /**
     * First-Strong Isolated
     */
    public static final char FSI = '\u2068';

    public static final char LRO = '\u202D';

    public static final char RLO = '\u202E';

    public static final String AR_LANG = "ar";
    public static final String BG_LANG = "bg";
    public static final String CS_LANG = "cs";
    public static final String FA_LANG = "fa";
    public static final String GU_LANG = "gu";
    public static final String MR_LANG = "mr";
    public static final String TH_LANG = "th";
    public static final String IW_LANG = "iw";
    public static final String PL_LANG = "pl";
    public static final String RU_LANG = "ru";
    public static final String UK_LANG = "uk";
    public static final String ZH_LANG = "zh";

    public static final String[] NUMBERS_0_31_WESTERN_ARABIC = {
            "0",
            "1",  "2",  "3",  "4",  "5",
            "6",  "7",  "8",  "9",  "10",
            "11", "12", "13", "14", "15",
            "16", "17", "18", "19", "20",
            "21", "22", "23", "24", "25",
            "26", "27", "28", "29", "30",
            "31"
    };
    public static final String[] NUMBERS_0_31_EASTERN_ARABIC = {
            "٠",
            "١",  "٢",  "٣",  "٤",  "٥",
            "٦",  "٧",  "٨",  "٩",  "١٠",
            "١١", "١٢", "١٣", "١٤", "١٥",
            "١٦", "١٧", "١٨", "١٩", "٢٠",
            "٢١", "٢٢", "٢٣", "٢٤", "٢٥",
            "٢٦", "٢٧", "٢٨", "٢٩", "٣٠",
            "٣١"
    };
    public static final String[] NUMBERS_0_31_PERSO_ARABIC = {
            "۰",
            "۱",  "۲",  "۳",  "۴",  "۵",
            "۶",  "۷",  "۸",  "۹",  "۱۰",
            "۱۱", "۱۲", "۱۳", "۱۴", "۱۵",
            "۱۶", "۱۷", "۱۸", "۱۹", "۲۰",
            "۲۱", "۲۲", "۲۳", "۲۴", "۲۵",
            "۲۶", "۲۷", "۲۸", "۲۹", "۳۰",
            "۳۱"
    };
    public static final String[] NUMBERS_0_31_GUJARATI = {
            "૦",
            "૧",  "૨",  "૩",  "૪",  "૫",
            "૬",  "૭",  "૮",  "૯",  "૧૦",
            "૧૧", "૧૨", "૧૩", "૧૪", "૧૫",
            "૧૬", "૧૭", "૧૮", "૧૯", "૨૦",
            "૨૧", "૨૨", "૨૩", "૨૪", "૨૫",
            "૨૬", "૨૭", "૨૮", "૨૯", "૩૦",
            "૩૧"
    };
    public static final String[] NUMBERS_0_31_DEVANAGARI = {
            "०",
            "१",  "२",  "३",  "४",  "५",
            "६",  "७",  "८",  "९",  "१०",
            "११", "१२", "१३", "१४", "१५",
            "१६", "१७", "१८", "१९", "२०",
            "२१", "२२", "२३", "२४", "२५",
            "२६", "२७", "२८", "२९", "३०",
            "३१"
    };
    public static final String[] NUMBERS_0_31_THAI = {
            "๐",
            "๑",  "๒",  "๓",  "๔",  "๕",
            "๖",  "๗",  "๘",  "๙",  "๑๐",
            "๑๑", "๑๒", "๑๓", "๑๔", "๑๕",
            "๑๖", "๑๗", "๑๘", "๑๙", "๒๐",
            "๒๑", "๒๒", "๒๓", "๒๔", "๒๕",
            "๒๖", "๒๗", "๒๘", "๒๙", "๓๐",
            "๓๑"
    };

    public static final char[] DIGITS_WESTERN_ARABIC = {
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'
    };
    public static final char[] DIGITS_EASTERN_ARABIC = {
            '٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩'
    };
    public static final char[] DIGITS_PERSO_ARABIC = {
            '۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'
    };
    public static final char[] DIGITS_GUJARATI = {
            '૦', '૧', '૨', '૩', '૪', '૫', '૬', '૭', '૮', '૯'
    };
    public static final char[] DIGITS_DEVANAGARI = {
            '०', '१', '२', '३', '४', '५', '६', '७', '८', '९'
    };
    public static final char[] DIGITS_THAI = {
            '๐', '๑', '๒', '๓', '๔', '๕', '๖', '๗', '๘', '๙'
    };

    public static final int WESTERN_ARABIC = 0;
    public static final int EASTERN_ARABIC = 1;
    public static final int PERSO_ARABIC = 2;
    public static final int GUJARATI = 3;
    public static final int DEVANAGARI = 4;
    public static final int THAI = 5;
    @IntDef({WESTERN_ARABIC, EASTERN_ARABIC, PERSO_ARABIC, GUJARATI, DEVANAGARI, THAI})
    @Retention(RetentionPolicy.SOURCE)
    public @interface NumeralSystem {}


    public static final int AM = 0;
    public static final int PM = 1;
    @IntDef({AM, PM})
    @Retention(RetentionPolicy.SOURCE)
    public @interface TimeFormat {}

    // Typefaces
    public static final String ROBOTO_C_BOLD_FONT = "RobotoCondensed-Bold";
    public static final String ROBOTO_C_LIGHT_FONT = "RobotoCondensed-Light";
    public static final String ROBOTO_MEDIUM_FONT = "Roboto-Medium";

    private static final SimpleArrayMap<String, Typeface> sCache = new SimpleArrayMap<>();
    private static final StringBuilder sBuilder = new StringBuilder();
    private static final StringBuilder sFmtBuilder = new StringBuilder(100);

    /*
    Each call to Typeface.createFromAsset will load a new instance of the typeface into memory,
    and this memory is not consistently get garbage collected
    http://code.google.com/p/android/issues/detail?id=9904
    (It states released but even on Lollipop you can see the typefaces accumulate even after
    multiple GC passes)
    You can detect this by running:
    adb shell dumpsys meminfo com.your.packagenage
    You will see output like:
     Asset Allocations
        zip:/data/app/com.your.packagenage-1.apk:/assets/Roboto-Medium.ttf: 125K
        zip:/data/app/com.your.packagenage-1.apk:/assets/Roboto-Medium.ttf: 125K
        zip:/data/app/com.your.packagenage-1.apk:/assets/Roboto-Medium.ttf: 125K
        zip:/data/app/com.your.packagenage-1.apk:/assets/Roboto-Regular.ttf: 123K
        zip:/data/app/com.your.packagenage-1.apk:/assets/Roboto-Medium.ttf: 125K
    */
    public static Typeface getTypeface(final Context ctx, final String name) {
        synchronized (sCache) {
            if (!sCache.containsKey(name)) {
                sBuilder.setLength(0);
                final Typeface t = Typeface.createFromAsset(
                        ctx.getAssets(),
                        sBuilder.append("fonts/").append(name).append(".ttf").toString());
                sCache.put(name, t);
                return t;
            }
            return sCache.get(name);
        }
    }

    public static boolean isLayoutRtl(final Resources res) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            return res.getConfiguration().getLayoutDirection() == View.LAYOUT_DIRECTION_RTL;
        } else {
            switch (Locale.getDefault().getLanguage()) {
                case AR_LANG:
                case FA_LANG:
                case IW_LANG:
                    return true;

                default:
                    return false;
            }
        }
    }

    public static void setBackgroundCompat(final View v, final Drawable drawable) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            v.setBackground(drawable);
        } else {
            v.setBackgroundDrawable(drawable);
        }
    }

    @SuppressWarnings("deprecation")
    public static void setTextAppearanceCompat(@NonNull final TextView textView, @StyleRes final int resId) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            textView.setTextAppearance(resId);
        } else {
            textView.setTextAppearance(textView.getContext(), resId);
        }
    }

    @SuppressWarnings("deprecation")
    public static CharSequence fromHtmlCompat(final String text) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            return Html.fromHtml(text, Html.FROM_HTML_MODE_LEGACY);
        } else {
            return Html.fromHtml(text);
        }
    }

//--------------------------------------------------------------------------------------------------
//  Screen methods
//--------------------------------------------------------------------------------------------------

    public static float dpToPx(final float dp, final Resources res) {
        return dp * ((float) res.getDisplayMetrics().densityDpi / DisplayMetrics.DENSITY_DEFAULT);
    }

    public static int dpToPx(final int dp, final Resources res) {
        return (int) dpToPx((float) dp, res);
    }

    public static int spToPx(final float sp, final Resources res) {
        return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, sp, res.getDisplayMetrics());
    }

    public static float pxToDp(final float px, final Resources res) {
        return px / ((float) res.getDisplayMetrics().densityDpi / DisplayMetrics.DENSITY_DEFAULT);
    }

    public static int getScreenSize(final Resources res) {
        return res.getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK;
    }

    public static boolean isLargeScreen(final Resources res) {
        final int screenSize = getScreenSize(res);
        return screenSize == Configuration.SCREENLAYOUT_SIZE_LARGE
                || screenSize == Configuration.SCREENLAYOUT_SIZE_XLARGE;
    }

    public static boolean isSmallScreen(final Resources res) {
        return getScreenSize(res) == Configuration.SCREENLAYOUT_SIZE_SMALL;
    }

    public static boolean isPortOrientation(final Resources res) {
        return res.getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
    }

    public static boolean isLandOrientation(final Resources res) {
        return res.getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE;
    }

    public static int getOrientation(final Resources res) {
        return res.getConfiguration().orientation;
    }

    public static float getScreenHeight(final Resources res) {
        final DisplayMetrics displayMetrics = res.getDisplayMetrics();
        return displayMetrics.heightPixels / displayMetrics.density;
    }

    public static float getScreenWidth(final Resources res) {
        final DisplayMetrics displayMetrics = res.getDisplayMetrics();
        return displayMetrics.widthPixels / displayMetrics.density;
    }

//--------------------------------------------------------------------------------------------------
//  Network methods
//--------------------------------------------------------------------------------------------------

    @SuppressLint("MissingPermission")
    /**
     * In order to use add permission to manifest:
     * <uses-permission android:name="android.permission.ACCESS_NETWORK_STATE" />
     */
    public static boolean isNetworkAvailable(final Context context) {
        final ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        if (cm == null) return false;

        final NetworkInfo netInfo = cm.getActiveNetworkInfo();
        return netInfo != null && netInfo.isAvailable() && netInfo.isConnected();
    }

//--------------------------------------------------------------------------------------------------
//  Resolve methods
//--------------------------------------------------------------------------------------------------

    public static boolean resolveBoolean(final Context context, @AttrRes final int attr) {
        return resolveBoolean(context, attr, false);
    }
    /**
     * Gets the required boolean value from the current context, if possible/available
     *
     * @param context  The context to use as reference for the boolean
     * @param attr     Attribute id to resolve
     * @param fallback Default value to return if no value is specified in theme
     * @return the boolean value from current theme
     */
    public static boolean resolveBoolean(final Context context, @AttrRes final int attr, final boolean fallback) {
        final TypedArray a = context.getTheme().obtainStyledAttributes(new int[]{attr});
        try {
            return a.getBoolean(0, fallback);
        } finally {
            a.recycle();
        }
    }

    public static Drawable resolveDrawable(final Context context, @AttrRes final int attr) {
        return resolveDrawable(context, attr, null);
    }
    private static Drawable resolveDrawable(final Context context, @AttrRes final int attr,
                                            @SuppressWarnings("SameParameterValue") final Drawable fallback) {
        final TypedArray a = context.getTheme().obtainStyledAttributes(new int[]{attr});
        try {
            Drawable d = a.getDrawable(0);
            if (d == null && fallback != null)
                d = fallback;
            return d;
        } finally {
            a.recycle();
        }
    }

    public static int resolveDimension(final Context context, @AttrRes final int attr) {
        return resolveDimension(context, attr, -1);
    }
    private static int resolveDimension(final Context context, @AttrRes final int attr, final int fallback) {
        final TypedArray a = context.getTheme().obtainStyledAttributes(new int[]{attr});
        try {
            return a.getDimensionPixelSize(0, fallback);
        } finally {
            a.recycle();
        }
    }

    public static int resolveColor(final Context context, @AttrRes final int attr) {
        return resolveColor(context, attr, 0);
    }
    public static int resolveColor(final Context context, @AttrRes final int attr, final int fallback) {
        final TypedArray a = context.getTheme().obtainStyledAttributes(new int[]{attr});
        try {
            return a.getColor(0, fallback);
        } finally {
            a.recycle();
        }
    }

    public static String resolveString(final Context context, @AttrRes final int attr) {
        final TypedValue v = new TypedValue();
        context.getTheme().resolveAttribute(attr, v, true);
        return (String) v.string;
    }

//--------------------------------------------------------------------------------------------------
//  Formatters
//--------------------------------------------------------------------------------------------------

    private static char[] getDigits(final int numSystem) {
        switch (numSystem) {
            default:
            case WESTERN_ARABIC:    return DIGITS_WESTERN_ARABIC;
            case EASTERN_ARABIC:    return DIGITS_EASTERN_ARABIC;
            case PERSO_ARABIC:      return DIGITS_PERSO_ARABIC;
            case GUJARATI:          return DIGITS_GUJARATI;
            case DEVANAGARI:        return DIGITS_DEVANAGARI;
            case THAI:              return DIGITS_THAI;
        }
    }

    private static String[] getNumbers31(final int numSystem) {
        switch (numSystem) {
            default:
            case WESTERN_ARABIC:    return NUMBERS_0_31_WESTERN_ARABIC;
            case EASTERN_ARABIC:    return NUMBERS_0_31_EASTERN_ARABIC;
            case PERSO_ARABIC:      return NUMBERS_0_31_PERSO_ARABIC;
            case GUJARATI:          return NUMBERS_0_31_GUJARATI;
            case DEVANAGARI:        return NUMBERS_0_31_DEVANAGARI;
            case THAI:              return NUMBERS_0_31_THAI;
        }
    }

    public static String formatNumber31(final int number, final int numSystem) {
        return getNumbers31(numSystem)[number];
    }

    /**
     * Converts only from western arabic
     */
    public static void convertNumbers(final StringBuilder b, final int numSystem) {
        final int count = b.length();
        if (count == 0 || numSystem == WESTERN_ARABIC) return;
        final char[] digits = getDigits(numSystem);

        char currChar;
        for (int i = 0; i < count; i++) {
            currChar = b.charAt(i);
            if (currChar >= ZERO_DIGIT && currChar <= NINE_DIGIT) {
                b.setCharAt(i, digits[currChar - ZERO_DIGIT]);
            }
        }
    }

    public static CharSequence convertNumbers(final CharSequence text, final int numSystem) {
        if (numSystem == WESTERN_ARABIC) return text;

        sFmtBuilder.setLength(0);
        sFmtBuilder.append(text);
        convertNumbers(sFmtBuilder, numSystem);
        return sFmtBuilder.toString();
    }

    public static CharSequence[] convertNumbers(final CharSequence[] arr, final int numSystem) {
        if (numSystem == WESTERN_ARABIC) return arr;

        for (int i = arr.length - 1; i >= 0; i--) {
            arr[i] = convertNumbers(arr[i], numSystem);
        }
        return arr;
    }

    public static String formatInt(final int value, final int numSystem) {
        sFmtBuilder.setLength(0);
        sFmtBuilder.append(value);

        convertNumbers(sFmtBuilder, numSystem);
        return sFmtBuilder.toString();
    }

    public static String formatLong(final long value, final int numSystem) {
        sFmtBuilder.setLength(0);
        sFmtBuilder.append(value);

        convertNumbers(sFmtBuilder, numSystem);
        return sFmtBuilder.toString();
    }

    public static String formatIntTwoDig(final int value) {
        return formatIntTwoDig(value, WESTERN_ARABIC);
    }

    public static String formatIntTwoDig(final int value, final int numSystem) {
        sFmtBuilder.setLength(0);

        if (value <= 0) {
            sFmtBuilder.append(ZERO_DIGIT)
                       .append(ZERO_DIGIT);
        } else if (value < 10) {
            sFmtBuilder.append(ZERO_DIGIT)
                       .append(value);
        } else if (value >= 10) {
            sFmtBuilder.append(value);
        }

        convertNumbers(sFmtBuilder, numSystem);
        return sFmtBuilder.toString();
    }

    public static void clearFormatter(final Formatter f) {
        ((StringBuilder) f.out()).setLength(0);
    }

    @SuppressWarnings("NewApi")
    public static String getBestDateTimePattern(final Locale locale, final String skeleton) {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2
                ? DateFormat.getBestDateTimePattern(locale, skeleton)
                : skeleton;
    }
}