/*
 * @(#) SpeechRecognitionListener.java 2015. 1.
 *
 * Copyright 2015 Naver Corp. All rights Reserved.
 * Naver PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.naver.speech.clientapi;

import com.naver.speech.clientapi.SpeechConfig.EndPointDetectType;

/**
 * {@link SpeechRecognizer} 클래스에서 발생한 이벤트를 감지하여 해당하는 콜벡 메소드를 호출합니다.
 * 커스텀 리스너는 선언되어 있는 콜백 메소드를 오버라이딩(implements)하여, 애플리케이션에서 원하는 동작을 하도록 구현합니다.
 * <br>
 * 커스텀 리스너를 오버라이딩 한 후, 반드시 {@link SpeechRecognizer#setSpeechRecognitionListener(SpeechRecognitionListener)} 메소드를 통해 등록해야 합니다.
 */
public interface SpeechRecognitionListener {

	/**
	 * 음성인식이 완료된 상태일 때 호출됩니다.
	 */
	public void onInactive();

	/**
	 * 음성인식 준비가 완료된 상태일 때 호출됩니다.
	 */
	public void onReady();

	/**
	 * 음성 입력을 받은 즉시 호출됩니다.
	 * @param speech 음성 신호가 저장된 버퍼
	 */
	public void onRecord(short[] speech);

	/**
	 * 음성인식 서버로부터 인식 중간 결과를 받으면 호출됩니다. 중간 결과는 없거나 여러번 있을 수 있습니다.
	 * @param partialResult 음성인식 중간 결과
	 */
	public void onPartialResult(String partialResult);

	/**
	 * EPD(end point detection) 즉, 음성 발성이 종료되었음을 클라이언트가 감지하였을 때 호출됩니다.
	 * EPD는 {@link SpeechRecognizer#recognize(SpeechConfig)}를 호출할 때 전달된 {@link SpeechConfig}에 따라
	 * 음성인식 서버에 의해서 감지되기도 하고, 유저가 {@link SpeechRecognizer#stop()}을 호출하여 즉시 생성되기도 합니다.
	 * <br>
	 * 이렇게 EPD가 감지되면, 해당 콜백 함수가 호출됩니다.
	 */
	public void onEndPointDetected();

	/**
	 * 음성인식 서버로부터 인식 최종 결과를 받으면 호출됩니다.
	 * @param finalResult 음성인식 최종 결과
	 * <dl>
	 * <dt>{@link SpeechRecognitionResult#getResults()}</dt>
	 * <dd>
	 * 음성인식 최종 결과를 반환합니다. 신뢰도가 높은 순서대로 최대 5개의 후보를 반환합니다.
	 * </dd>
	 * </dl>
	 */
	public void onResult(SpeechRecognitionResult finalResult);

	/**
	 * 음성인식중에 오류가 발생하였을 때 호출됩니다.
	 * @param errorCode {@link SpeechRecognizer}에 정의된 오류 코드 중 하나
	 * @see SpeechRecognizer#ERROR_AUDIO_FINIALIZE
	 * @see SpeechRecognizer#ERROR_AUDIO_INITIALIZE
	 * @see SpeechRecognizer#ERROR_AUDIO_RECORD
	 * @see SpeechRecognizer#ERROR_NETWORK_FINALIZE
	 * @see SpeechRecognizer#ERROR_NETWORK_INITIALIZE
	 * @see SpeechRecognizer#ERROR_NETWORK_NACK
	 * @see SpeechRecognizer#ERROR_NETWORK_READ
	 * @see SpeechRecognizer#ERROR_NETWORK_WRITE
	 * @see SpeechRecognizer#ERROR_INVALID_RESULT
	 * @see SpeechRecognizer#ERROR_NO_CLIENT_RUNNING
	 * @see SpeechRecognizer#ERROR_INVALID_PACKET
	 * @see SpeechRecognizer#ERROR_SECURITY
	 * @see SpeechRecognizer#ERROR_TIMEOUT
	 * @see SpeechRecognizer#ERROR_UNKOWN_EVENT
	 * @see SpeechRecognizer#ERROR_VERSION
	 * @see SpeechRecognizer#ERROR_CLIENTINFO
	 * @see SpeechRecognizer#ERROR_SERVER_POOL
	 * @see SpeechRecognizer#ERROR_SESSION_EXPIRED
	 * @see SpeechRecognizer#ERROR_SPEECH_SIZE_EXCEEDED
	 * @see SpeechRecognizer#ERROR_EXCEED_TIME_LIMIT
	 * @see SpeechRecognizer#ERROR_WRONG_SERVICE_TYPE
	 * @see SpeechRecognizer#ERROR_WRONG_LANGUAGE_TYPE
	 * @see SpeechRecognizer#ERROR_OPENAPI_AUTH
	 * @see SpeechRecognizer#ERROR_QUOTA_OVERFLOW
	 */
	public void onError(int errorCode);

	/**
	 * hybrid EPD 모드에서 EPD 종류가 결정되면 호출됩니다.
	 * EPD 종류는 다음 두 가지 상황에 의해 결정됩니다.
	 * <dl>
	 * <dt>explicitly selected</dt>
	 * <dd>
	 * {@link SpeechRecognizer#selectEPDTypeInHybrid(EndPointDetectType)}가 호출됨으로써 결정됩니다.
	 * 유저가 직접 AUTO, MANUAL을 선택하는 경우가 여기에 해당됩니다. 이 함수가 정상적으로 동작했을 때에만 onEndPointDetectTypeSelected 콜백 함수가 호출됩니다.
	 * </dd>
	 * <dt>implicitly selected</dt>
	 * <dd>
	 * 약 600ms 동안 {@link SpeechRecognizer#selectEPDTypeInHybrid(EndPointDetectType)}가 호출되지 않으면, 자동으로 결정됩니다.
	 * 이 경우에는 MANUAL로 결정됩니다.
	 * </dd>
	 * </dl>
	 * @param epdType 결정된 EPD 종류
	 */
	public void onEndPointDetectTypeSelected(EndPointDetectType epdType);
}
