/*
 * @(#) SpeechConfig.java 2015. 1.
 *
 * Copyright 2015 Naver Corp. All rights Reserved.
 * Naver PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.naver.speech.clientapi;

import java.util.Locale;

/**
 * 이 클래스는 네이버 음성인식에서 제공하는 음성인식기에 대한 설정을 포함하고 있습니다.
 * <dl>
 * <dt>언어</dt>
 * <dd>한국어, 영어, 일본어, 중국어 총 4가지 언어를 제공합니다.</dd>
 * <dt>EPD(end point detection) 모드</dt>
 * <dd>
 * AUTO, MANUAL, HYBRID 총 3가지 EPD 모드를 제공합니다.
 * <br>
 * AUTO : 서버에서 EPD를 감지하여 {@link SpeechRecognizer#onEndPointDetected()}를 호출합니다.
 * <br>
 * MANUAL : 유저가 직접 {@link SpeechRecognizer#stop()}을 통해서 {@link SpeechRecognizer#onEndPointDetected()}를 호출합니다.
 * <br>
 * HYBRID : 상황에 따라 AUTO 또는 MANUAL로 동작합니다.
 * 이는 유저가 {@link SpeechRecognizer#selectEPDTypeInHybrid(EndPointDetectType)}를 호출하여 결정되거나, 또는 내부적으로 implicit하게 결정됩니다.
 * </dd>
 * </dl>
 */
public class SpeechConfig {

	public static enum ServiceType {
		WEBSEARCH	(0),
		MAPSEARCH	(1),
		TRANSTALK	(2),
		SHOPPING	(3),
		OPENAPI	(4),
		JUNIVER	(5),
		LIVESEARCH	(6),
		MAPNAVI		(7),
		PAPAGO		(9);

		private int serviceType;
		private ServiceType(int serviceType) {
			this.serviceType = serviceType;
		}
		int toInteger() { return this.serviceType; }
	}

	public static enum LanguageType {
		KOREAN		(0),
		JAPANESE	(1),
		ENGLISH		(2),
		CHINESE		(3);

		private int languageType;
		private LanguageType(int languageType) {
			this.languageType = languageType;
		}
		int toInteger() {
			return this.languageType;
		}
	}

	public static enum EndPointDetectType {
		AUTO(0),
		MANUAL(1),
		HYBRID(2);

		private int epdType;
		EndPointDetectType(int epdType) {
			this.epdType = epdType;
		}
		public int toInteger() {
			return this.epdType;
		}
	}

	private ServiceType			mServiceType;
	private LanguageType		mLanguageType;
	private EndPointDetectType	mEndPointDetectType;
	private boolean				mQuestionDetection;
	private String				mExtraInfo;

	public SpeechConfig(
			ServiceType serviceType,
			LanguageType languageType,
			boolean questionDetection,
			EndPointDetectType endPointDetectType)
			{
		this.mServiceType		= serviceType;
		this.mLanguageType		= languageType;
		this.mQuestionDetection	= questionDetection;
		this.mEndPointDetectType	= endPointDetectType;
		this.mExtraInfo			= null;
			}

	public SpeechConfig(
			ServiceType serviceType,
			LanguageType languageType,
			boolean questionDetection,
			EndPointDetectType endPointDetectType,
			String server,
			int port)
			{
		this.mServiceType		= serviceType;
		this.mLanguageType		= languageType;
		this.mQuestionDetection	= questionDetection;
		this.mEndPointDetectType	= endPointDetectType;
		this.mExtraInfo			= String.format(Locale.ENGLISH, "%s:%d", server, port);
			}

	/////////////////////////////////////////////////////////////////
	//
	// setter / getter
	//
	public ServiceType getServiceType() {
		return mServiceType;
	}

	public void setServiceType(ServiceType serviceType) {
		this.mServiceType = serviceType;
	}

	public LanguageType getLanguageType() {
		return mLanguageType;
	}

	public void setLanguageType(LanguageType languageType) {
		this.mLanguageType = languageType;
	}

	public EndPointDetectType getEndPointDetectType() {
		return mEndPointDetectType;
	}

	public void setEndPointDetectType(EndPointDetectType endPointDetectType) {
		this.mEndPointDetectType = endPointDetectType;
	}

	public boolean isQuestionDetection() {
		return mQuestionDetection;
	}

	public void setQuestionDetection(boolean questionDetection) {
		this.mQuestionDetection = questionDetection;
	}

	public String getExtraInfo() { return mExtraInfo; }

	public void setExtraInfo(String server, int port) {
		this.mExtraInfo = String.format(Locale.ENGLISH, "%s:%d", server, port);
	}
}
