(ns couchdb.client
  (:require 
    [couchdb.errors]
    [clj-http.core :as http-core]
    [clj-http.client :as http-client])
  (:use 
    [slingshot.slingshot :only [try+ throw+ get-throw-context get-thrown-object]] 
    [clojure.string :only [join]]
    [cheshire.core :only [generate-string parse-string ]])
  (:import 
    (couchdb.errors InvalidDatabaseName DatabaseNotFound DocumentNotFound AttachmentNotFound ResourceConflict PreconditionFailed ServerError)
    (java.net URLEncoder)))


(def #^{:doc
        "Executes the HTTP request corresponding to the given map and returns the
        response map for corresponding to the resulting HTTP response.

        In addition to the standard Ring request keys, the following keys are also
        recognized:
        * :url
        * :method
        * :query-params
        * :basic-auth
        * :content-type
        * :accept
        * :accept-encoding
        * :as

        The following additional behaviors over also automatically enabled:
        * No exceptions are thrown for status codes
        * Gzip and deflate responses are accepted and decompressed
        * Input and output bodies are coerced as required and indicated by the :as
        option."}
  http-request
  (-> #'http-core/request
    http-client/wrap-redirects
    http-client/wrap-decompression
    http-client/wrap-input-coercion
    http-client/wrap-output-coercion
    http-client/wrap-query-params
    http-client/wrap-basic-auth
    http-client/wrap-accept
    http-client/wrap-accept-encoding
    http-client/wrap-content-type
    http-client/wrap-method
    http-client/wrap-url))

(defn read-response [response]
  (try 
    (assoc response :json (parse-string (apply str (:body response)) true))
    ;; if there's an error reading the JSON, just don't make a :json key
    (catch Exception e response))) 

(defn couch-request* [response]
  (let [result (read-response response)]
    (if (>= (:status result) 400)
      (throw+ 
        (condp = (:status result)
          404 (condp = (:reason (:json result))
                "no_db_file" (DatabaseNotFound. (:json result))   
                "Document is missing attachment" (AttachmentNotFound. (:json result))
                (DocumentNotFound. (:json result)))

          409 (ResourceConflict. (:json result))
          412 (PreconditionFailed. (:json result))
          (ServerError. (:json result))))
      result)))

(defn couch-request [args]
  (couch-request* (http-request (merge {:method :get} args))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;         Utilities           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn url-encode
  "Wrapper around java.net.URLEncoder returning a (UTF-8) URL encoded
  representation of argument, either a string or map."
  [arg]
  (if (map? arg)
    (join \& (map #(join \= (map url-encode %)) arg))
    (URLEncoder/encode (name arg) "UTF-8")))

(defn valid-dbname?
  [database]
  (boolean (re-find #"^[a-z][a-z0-9_$()+-/]*$" database)))

(defn validate-dbname
  [database]
  (if (valid-dbname? database)
    (url-encode database)
    (throw+ (InvalidDatabaseName. database))))

(defn stringify-top-level-keys
  [[k v]]
  (if (keyword? k)
    [(if-let [n (namespace k)]
       (str n (name k))
       (name k))
     v]
    [k v]))

(defn- normalize-url
  "If not present, appends a / to the url-string."
  [url]
  (if-not (= (last url) \/)
    (str url \/ )
    url))

(defn- vals-lift [f m]
  (reduce (fn [acc [k v]] (assoc acc k (f v))) {} (seq m)))

(def #^{:private true} vals2json (partial vals-lift generate-string))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;       Utility Macros       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- vars-to-rebind []
  (letfn [(rebind? [var]
            (let [m (meta var)]
              (and (= (:ns m) (find-ns 'couchdb.client)) (:dynamic m))))]
    (filter rebind? (vals (ns-map *ns*)))))

(defmacro with-server
  "with-server rebinds all couchdb functions which take a server argument with
  the first argument, so you can call the functions without the server argument.

  Example:
  ;(with-server http://localhost:5984
  ;  (database-list))"

  [server & body]
  (let [ssharp (gensym "server-")]      ;necessary because nested-`
    `(let [~ssharp ~server]
       (with-bindings ~(apply hash-map
                              (mapcat #(vector % `(partial (var-get ~%) ~ssharp))
                                      (vars-to-rebind)))
         (do
           ~@body)))))

;; (with-server (apply str (concat "http://" "localhost" ":5984")) (database-list))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;          Databases          ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn ^:dynamic database-list [server]
  (:json (couch-request {:url (str (normalize-url server) "_all_dbs"), :method :get})))

(defn ^:dynamic database-create [server database]  
  (when-let [database (validate-dbname database)]
    (couch-request {:url (str (normalize-url server) database),
                    :method :put})
    database))

(defn ^:dynamic database-delete [server database]
  (when-let [database (validate-dbname database)]
    (couch-request {:url (str (normalize-url server) database), :method :delete}) true))

(defn ^:dynamic database-info [server database]
  (when-let [database (validate-dbname database)]
    (:json (couch-request {:url (str (normalize-url server) database),
                           :method :get}))))

(defn ^:dynamic database-compact
  [server database]
  (when-let [database (validate-dbname database)]
    (couch-request {:url (str (normalize-url server) database "/_compact"),
                    :content-type :json
                    :method :post})
    true))


(defn ^:dynamic database-replicate [src-server src-database target-server target-database]
  (couch-request {:url (str (normalize-url target-server) "_replicate")  :method :post 
                  :content-type :json 
                  :body (generate-string {"source" src-database, "target" target-database})}))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;         Documents           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(declare document-get)

(defn- do-get-doc
  [database document]
  (if (map? document)
    (if-let [id (:_id document)]
      id
      (throw+ (ResourceConflict. "missing :_id key"))) document))

(defn- do-get-rev
  [server database document]
  (if (map? document)
    (if-let [rev (:_rev document)]
      rev
      (throw+ (ResourceConflict. "missing :_rev key")))
    (:_rev (document-get (normalize-url server) database document))))

(defn- do-document-touch
  [server database payload id method]
  (when-let [database (validate-dbname database)]
    (let [response (:json (couch-request {:url (str (normalize-url server)
                                                    database
                                                    (when id (str "/" (url-encode (name id))))),
                                          :method method
                                          :content-type :json
                                          :body (generate-string payload)}))]					       
      (merge payload
             {:_id (:id response)
              :_rev (:rev response)}))))

(defn ^:dynamic document-list
  ([server database]
   (when-let [database (validate-dbname database)]
     (map :id (:rows 
                (:json (couch-request {:url (str (normalize-url server)
                                                 database
                                                 "/_all_docs")}))))))
  ([server database & [options]]
   (when-let [database (validate-dbname database)] 
     (map (if (:include_docs options) :doc :id)
          (:rows (:json (couch-request {:url (str (normalize-url server) database "/_all_docs?"
                                                  (url-encode (vals2json options)))})))))))

(defn ^:dynamic document-create
  ([server database payload]
   (do-document-touch (normalize-url server) database payload nil :post))
  ([server database id payload]
   (do-document-touch (normalize-url server) database payload id :put)))

(defn ^:dynamic document-update
  [server database id payload]
  ;(assert (:_rev payload)) ;; payload needs to have a revision or you'll get a PreconditionFailed error
  (let [id (do-get-doc database id)]
    (do-document-touch (normalize-url server) database payload id :put)))

(defn ^:dynamic document-get
  ([server database id]
   (when-let [database (validate-dbname database)]
     (let [id (do-get-doc database id)]
       (:json (couch-request {:url (str (normalize-url server) database "/" (name id))})))))

  ([server database id rev]
   (when-let [database (validate-dbname database)]
     (let [id (do-get-doc database id)]
       (:json (couch-request {:url (str (normalize-url server) database "/"
                                        (url-encode (name id)) "?rev=" rev)}))))))

(defn ^:dynamic document-delete
  ([server database id rev]
   (if-not (empty? id)
     (when-let [database (validate-dbname database)]
       (let [id (do-get-doc database id)]
         (couch-request {:url (str (normalize-url server) database "/"
                                   (url-encode (name id)) "?rev=" rev),
                         :method :delete})
         true))
     false))
  ([server database id]
   (if-not (empty? id)
     (when-let [database (validate-dbname database)]
       (let [id (do-get-doc database id)]
         (document-delete server database id
                          (do-get-rev (normalize-url server)
                                      database id))))
     false)))

(defn ^:dynamic document-bulk-update
  "Does a bulk-update to couchdb, accoding to
  http://wiki.apache.org/couchdb/HTTP_Bulk_Document_API"
  [server database document-coll & [request-options]]
  (when-let [database (validate-dbname database)]
    (let [response (:json
                     (couch-request {:url (str (normalize-url server) database "/_bulk_docs"
                                               (url-encode (vals2json request-options))),
                                     :method :post,
                                     :content-type :json,
                                     :body (generate-string {:docs document-coll})}))]
      ;; I don't know if this is correct... I assume that the server sends the
      ;; ids and revs in the same order as ib my request back.
      (map (fn [respdoc, orgdoc]
             (merge orgdoc
                    {:_id (:id respdoc)
                     :_rev (:rev respdoc)}))
           response document-coll))))

(defn- revision-comparator
  [x y]
  (> (Integer/decode (apply str (take-while #(not= % \-) x)))
     (Integer/decode (apply str (take-while #(not= % \-) y)))))

(defn ^:dynamic document-revisions
  [server database id]
  (when-let [database (validate-dbname database)]
    (let [id (do-get-doc database id)]
      (apply merge (map (fn [m]
                          (sorted-map-by revision-comparator (:rev m) (:status m)))
                        (:_revs_info (:json
                                       (couch-request {:url (str (normalize-url server) database "/"
                                                                 (url-encode (name id)) "?revs_info=true")}))))))))

(defn- url-encode-str [s]
  (-> s
    name
    url-encode))

(defn ^:dynamic document-get-conflicts
  "Returns a list of document revisions that conflict with the doc id"
  [server database id]
  (when-let [database (validate-dbname database)]
    (:_conflicts (:json (couch-request {:url (str (normalize-url server) database "/"
                                                  (url-encode-str (do-get-doc database id)) "?conflicts=true"),
                                        :content-type :json})))))



(defn ^:dynamic document-resolve-conflict
  "Function for resolving a conflicted document takes a server,
  database, document id, conflict revision (see document-get-conflicts)
  and a function that takes two args, the first conflicted doc, the second
  the current doc"
  [server database id conflict-rev resolve-fn]
  (when-let [database (validate-dbname database)]
    (let [merged-doc (resolve-fn (document-get server database id conflict-rev)
                                 (document-get server database id))]
      (document-update server database id merged-doc)
      (document-delete server database id conflict-rev))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;            Views            ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn make-couchdb-params [options & to-jsonify]
  "Return a URL-encoded parameter string of 'options', JSON-transforming those parameters in 'to-jsonify'"
  (let [jsonified-options (vals2json (select-keys options to-jsonify))
        regular-options (apply dissoc options to-jsonify)]
    (merge regular-options jsonified-options)))


(defn ^:dynamic view-list [server db design-doc]
  "Get a list of views, including their definitions"
  (:views (:json (couch-request {:url (str (normalize-url server) db "/_design/" design-doc) }))))


(defn  ^:dynamic  view-add [server db design-doc view-name type js]
  "Add a JavaScript view to a design document in a database. Type is one of :map or :reduce.
  Overwrites any previous view of the same name and type."
  (with-server server
    (let [doc-path (str "_design/" design-doc)]
      (try+
        (let [doc-content (document-get db doc-path)]
          (document-update db doc-path (assoc-in doc-content [:views view-name type] js)))
        (catch DocumentNotFound e
          (document-create db doc-path {:language "javascript" :views {(keyword view-name) {(keyword type) js}}}))))))


(defn ^:dynamic view-get [server db design-doc view-name & [view-options]]
  (:json (couch-request {:url (str (normalize-url server) db "/_design/" design-doc "/_view/" view-name "?"
                                   (url-encode (make-couchdb-params view-options :key :startkey :endkey)))})))

(defn ^:dynamic view-temp-get [server db view-map & [view-options]]
  (:json (couch-request {:url (str (normalize-url server) db "/_temp_view?"
                                   (url-encode (make-couchdb-params view-options :key :startkey :endkey)))
                         :method :post,
                         :content-type :json,
                         :body (generate-string view-map)})))

(defn view-create
  "Create a map/reduce view.  The new view is a clojure-map with keys of :map and
  (optionally) :reduce.  Values are the string representations of the functions
  in the language of the design-doc (usually javascript)."
  [server db design-name view-name view-map]
  (let [design-key (str "_design/" design-name)]
    (when-let [design-doc (document-get server db design-key)]
      (document-update server db design-key
                       (assoc design-doc :views (assoc (:views design-doc) (keyword view-name) view-map))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;        Attachments          ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn ^:dynamic attachment-list
  [server database document]
  (let [document (do-get-doc database document)]
    (into {} (map stringify-top-level-keys
                  (:_attachments (document-get (normalize-url server)
                                               database document))))))

(defn ^:dynamic attachment-create
  [server database document id payload content-type]
  (when-let [database (validate-dbname database)]
    (let [document (do-get-doc database document)
          rev (do-get-rev (normalize-url server) database document)]
      (couch-request {:url (str (normalize-url server) database "/"
                                (url-encode (name document)) "/"
                                (url-encode (name id)) "?rev=" rev),
                      :method :put,
                      :content-type content-type,
                      :body payload}))
    id))

(defn attachment-get
  "Returns the attachment in :body
  If the data is text/plain, :body is a seq of string, otherwise it is a byte array"
  [server database document id]
  (when-let [database (validate-dbname database)]
    (let [document (do-get-doc database document)
          response (couch-request {:url (str (normalize-url server) database "/"
                                             (url-encode (name document)) "/"
                                             (url-encode (name id)))
                                   :as :byte-array})
          content-type (.toLowerCase (get (:headers response) "content-type"))]
      {:body (if (.contains content-type "text/plain")
               (String. (:body response))
               (:body response))
       :content-type content-type})))


(defn attachment-delete
  [server database document id]
  (when-let [database (validate-dbname database)]
    (let [document (do-get-doc database document)
          rev (do-get-rev (normalize-url server) database document)]
      (couch-request {:url (str (normalize-url server) database "/"
                                (url-encode (name document)) "/"
                                (url-encode (name id)) "?rev=" rev),
                      :method :delete})
      true)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;            Shows            ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn ^:dynamic show-get
  "Returns the contents of a show as a list of strings according to
  http://wiki.apache.org/couchdb/Formatting_with_Show_and_List"
  [server database design-doc show-name id & [show-options]]
  (:body
    (couch-request {:url (str (normalize-url server) database "/_design/"
                              design-doc "/_show/" show-name "/"
                              id "?" (url-encode (vals2json show-options)))})))
