package com.mapbox.android.accounts.navigation.sku.v1;

import android.content.SharedPreferences;
import android.os.SystemClock;
import android.support.annotation.Keep;

import com.mapbox.android.accounts.v1.MapboxAccounts;

/**
 * The class is responsible to request @link[MapboxAccounts.java] to generate the session token
 * and rotate the SKU token based on the refresh time
 */
public class TripsSku implements SkuGenerator {

    enum RotateTripsType {
        INVALID,
        ROTATE_ON_TIMER_EXPIRE,
        ROTATE_ON_REQUEST_COUNT_EXPIRE
    }

    private static final String KEY_PREFERENCE_NAV_TRIPS_SKU = "com.mapbox.navigation.accounts.trips.skutoken";
    private static final String KEY_PREFERENCE_NAV_ROUTE_REQ_COUNT = "com.mapbox.navigation.accounts.trips.request.count";
    private static final String KEY_PREFERENCE_NAV_TRIPS_TIMESTAMP = "com.mapbox.navigation.accounts.trips.timestamp";
    private static final int DEFAULT_TRIP_REQUEST_COUNT = 1;
    private static final long DEFAULT_TRIP_TOKEN_TIMER = 0L;

    private boolean isNavigating = false;
    private SharedPreferences preferences;
    private long timerExpireAfter;
    private int routeRequestThreshold;

    @Keep
    public TripsSku(SharedPreferences preferences, long timerExpireAfter,int routeRequestThreshold) {
        this.preferences = preferences;
        this.timerExpireAfter = timerExpireAfter;
        this.routeRequestThreshold = routeRequestThreshold;
    }

    /**
     * Generates the SKU token
     * @return the SKU token
     */
    @Keep
    @Override
    public String generateToken() {
        refreshSkuToken();
        return retrieveTripsSkuToken();
    }

    /**
     * Sets the number of request made to DirectionsAPI count to the default value
     */
    @Keep
    @Override
    public void onNavigationStart() {
        this.isNavigating = true;
        setRouteRequestCount(DEFAULT_TRIP_REQUEST_COUNT);
    }

    /**
     * Sets the last time a request was made to DirectionsAPI to the default value
     */
    @Keep
    @Override
    public void onNavigationEnd() {
        this.isNavigating = false;
        setTimerExpiry(DEFAULT_TRIP_TOKEN_TIMER);
    }

    private void refreshSkuToken() {
        if (isNavigating) {
            handleRotationOnNavActive();
        } else {
            handleRotationOnNavInActive();
        }
    }

    private void handleRotationOnNavActive() {
        if (validateTimerExpiry()) {
            setTimerExpiry(getNow());
            persistTripsSkuToken();
        }
    }

    private void handleRotationOnNavInActive() {
        int requestCount;
        RotateTripsType rotationType = validateRotation();
        if (rotationType == RotateTripsType.ROTATE_ON_TIMER_EXPIRE
                || rotationType == RotateTripsType.ROTATE_ON_REQUEST_COUNT_EXPIRE) {
            requestCount = 1;
            setTimerExpiry(getNow());
            persistTripsSkuToken();
        } else {
            requestCount = getRouteRequestCountThreshold();
            requestCount++;
        }
        setRouteRequestCount(requestCount);
    }

    private RotateTripsType validateRotation() {
        boolean routeReqCountExpired = validateRouteRequestCountExpiry();
        boolean timerExpired = validateTimerExpiry();
        if (routeReqCountExpired) {
            return RotateTripsType.ROTATE_ON_REQUEST_COUNT_EXPIRE;
        } else if (timerExpired) {
            return RotateTripsType.ROTATE_ON_TIMER_EXPIRE;
        } else {
            return RotateTripsType.INVALID;
        }
    }

    private boolean validateTimerExpiry() {
        long skuTokenTimeStamp = getTimerExpiry();
        return isTwoHoursExpired(skuTokenTimeStamp);
    }

    private boolean validateRouteRequestCountExpiry() {
        int routeRequestCount = getRouteRequestCountThreshold();
        return routeRequestCount > routeRequestThreshold;
    }

    private void setRouteRequestCount(int count) {
        preferences.edit().putInt(KEY_PREFERENCE_NAV_ROUTE_REQ_COUNT, count).apply();
    }

    private int getRouteRequestCountThreshold() {
        return preferences.getInt(KEY_PREFERENCE_NAV_ROUTE_REQ_COUNT, DEFAULT_TRIP_REQUEST_COUNT);
    }

    private void persistTripsSkuToken() {
        String token = generateTripsSkuToken();
        preferences.edit().putString(KEY_PREFERENCE_NAV_TRIPS_SKU, token).apply();
    }

    private String retrieveTripsSkuToken() {
        return preferences.getString(KEY_PREFERENCE_NAV_TRIPS_SKU, "");
    }

    private String generateTripsSkuToken() {
        return MapboxAccounts.obtainNavigationSkuSessionToken();
    }

    private void setTimerExpiry(long then) {
        preferences.edit().putLong(KEY_PREFERENCE_NAV_TRIPS_TIMESTAMP, then).apply();
    }

    private long getTimerExpiry() {
        return preferences.getLong(KEY_PREFERENCE_NAV_TRIPS_TIMESTAMP, DEFAULT_TRIP_TOKEN_TIMER);
    }

    private boolean isTwoHoursExpired(long then) {
        return isExpired(getNow(), then);
    }

    private long getNow() {
        return SystemClock.elapsedRealtime();
    }

    private boolean isExpired(long now, long then) {
        return (now - then) / 1000 > timerExpireAfter;
    }
}
