package com.mapbox.android.accounts.navigation.sku.v1;

import android.content.SharedPreferences;
import android.os.SystemClock;
import android.support.annotation.Keep;
import android.text.TextUtils;

import com.mapbox.android.accounts.v1.MapboxAccounts;

import static com.mapbox.android.accounts.v1.AccountsConstants.KEY_PREFERENCE_MAPS_SKU_TOKEN;

/**
 * The class is responsible to request @link[MapboxAccounts.java] to generate the MAU token
 * and rotate the SKU token based on the refresh time
 */
public class MauSku implements SkuGenerator {

    private static final String KEY_PREFERENCE_NAV_MAU_SKU = "com.mapbox.navigation.accounts.mau.skutoken";
    private static final String KEY_PREFERENCE_NAV_USER_ID = "com.mapbox.navigation.accounts.mau.userid";
    private static final String KEY_PREFERENCE_NAV_MAU_TIMESTAMP = "com.mapbox.navigation.accounts.mau.timestamp";
    private static final long DEFAULT_TOKEN_TIMER = 0L;

    private SharedPreferences preferences;
    private long timerExpireAfter;

    @Keep
    public MauSku(SharedPreferences preferences, long timerExpireAfter) {
        this.preferences = preferences;
        this.timerExpireAfter = timerExpireAfter;
    }

    /**
     * Generates the SKU token
     * @return the SKU token
     */
    @Keep
    @Override
    public String generateToken() {
        refreshSkuToken();
        return retrieveMauSkuToken();
    }

    @Keep
    @Override
    public void onNavigationStart() {
    }

    /**
     * Sets the last time a request was made to DirectionsAPI to the default value
     */
    @Keep
    @Override
    public void onNavigationEnd() {
        setTimerExpiry(DEFAULT_TOKEN_TIMER);
    }

    private void refreshSkuToken() {
        if (!shouldRefreshSku()) {
            return;
        }
        setTimerExpiry(getNow());
        persistMauSkuToken();
        persistMapsSkuToken();
    }

    private boolean shouldRefreshSku() {
        return validateTimerExpiry();
    }

    private boolean validateTimerExpiry() {
        long skuTokenTimeStamp = getTimerExpiry();
        return isOneHoursExpired(skuTokenTimeStamp);
    }

    private void setTimerExpiry(long then) {
        preferences.edit().putLong(KEY_PREFERENCE_NAV_MAU_TIMESTAMP, then).apply();
    }

    private long getTimerExpiry() {
        return preferences.getLong(KEY_PREFERENCE_NAV_MAU_TIMESTAMP, DEFAULT_TOKEN_TIMER);
    }

    private void persistMauSkuToken() {
        String token = generateMauSkuToken();
        preferences.edit().putString(KEY_PREFERENCE_NAV_MAU_SKU, token).apply();
    }

    private String retrieveMauSkuToken() {
        return preferences.getString(KEY_PREFERENCE_NAV_MAU_SKU, "");
    }

    private void persistMapsSkuToken() {
        String mapsSkuToken = generateMapsSkuToken();
        preferences.edit().putString(KEY_PREFERENCE_MAPS_SKU_TOKEN, mapsSkuToken).apply();
    }

    private void persistMauUserId(String userId) {
        preferences.edit().putString(KEY_PREFERENCE_NAV_USER_ID, userId).apply();
    }

    private String retrieveUserId() {
        return preferences.getString(KEY_PREFERENCE_NAV_USER_ID, "");
    }

    private String getUserId() {
        String userId = retrieveUserId();
        if (TextUtils.isEmpty(userId)) {
            userId = generateUserId();
            persistMauUserId(userId);
        }
        return userId;
    }

    private String generateUserId() {
        return MapboxAccounts.obtainEndUserId();
    }

    private String generateMauSkuToken() {
        return MapboxAccounts.obtainNavigationSkuUserToken(getUserId());
    }

    private String generateMapsSkuToken() {
        return MapboxAccounts.obtainMapsSkuUserToken(getUserId());
    }

    private boolean isOneHoursExpired(long then) {
        return isExpired(getNow(), then);
    }

    private long getNow() {
        return SystemClock.elapsedRealtime();
    }

    private boolean isExpired(long now, long then) {
        return (now - then) / 1000 > timerExpireAfter;
    }
}
