package com.mapbox.mapboxsdk.plugins.china.maps;

import com.mapbox.mapboxsdk.maps.Style;
import com.mapbox.mapboxsdk.plugins.china.constants.ChinaStyle;
import com.mapbox.mapboxsdk.storage.FileSource;
import com.mapbox.mapboxsdk.storage.Resource;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Class used for automatically mapping styles to {@link ChinaStyle} equivalents.
 */
public class ChinaStyleMapper implements FileSource.ResourceTransformCallback {
  /**
   * Currently we have 3 China styles
   */
  private static final int STYLE_COUNT = 3;
  /**
   * The array to represent the global styles' URLs
   */
  static String[] originalSuffix = new String[STYLE_COUNT];
  /**
   * The array to represent the China styles' URLs
   */
  static String[] chinaSuffix = new String[STYLE_COUNT];

  /* Split suffixes from style url, so we can use them in the following transform.*/
  static {
    originalSuffix[0] = Style.MAPBOX_STREETS.split("styles/")[1];
    chinaSuffix[0] = ChinaStyle.MAPBOX_STREETS_CHINESE.split("styles/")[1];
    originalSuffix[1] = Style.LIGHT.split("styles/")[1];
    chinaSuffix[1] = ChinaStyle.MAPBOX_LIGHT_CHINESE.split("styles/")[1];
    originalSuffix[2] = Style.DARK.split("styles/")[1];
    chinaSuffix[2] = ChinaStyle.MAPBOX_DARK_CHINESE.split("styles/")[1];
  }

  @Override
  public String onURL(int kind, String url) {
    // Only dark style URLs
    if (kind == Resource.STYLE) {
      // Do nothing if the URL already is a China style
      if (url.contains("zh-v")) {
        return url;
      }
      // For these three styles, mapping them to corresponding China styles
      for (int i = 0; i < STYLE_COUNT; i++) {
        if (url.contains(originalSuffix[i])) {
          return url.replace(originalSuffix[i], chinaSuffix[i]);
        }
      }
      // For other styles, map them to the default MAPBOX_STREETS_CHINESE style
      Pattern pattern = Pattern.compile("v1/.*\\?");
      Matcher matcher = pattern.matcher(url);
      return matcher.replaceAll("v1/" + chinaSuffix[0] + "?");
    }
    return url;
  }
}
