package com.mapbox.mapboxsdk.plugins.china.maps;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.UiThread;
import android.util.AttributeSet;

import com.mapbox.geojson.shifter.CoordinateShifter;
import com.mapbox.geojson.shifter.CoordinateShifterManager;
import com.mapbox.mapboxsdk.maps.MapView;
import com.mapbox.mapboxsdk.maps.MapboxMapOptions;
import com.mapbox.mapboxsdk.plugins.china.constants.MapboxChinaConstants;
import com.mapbox.mapboxsdk.plugins.china.shift.ChinaCoordinateShifter;
import com.mapbox.mapboxsdk.storage.FileSource;

/**
 * An extension of the Map SDK's MapView class. This class automatically handles logic related to the display
 * of China map tiles and data which uses China's GCJ-02 coordinate system.
 *
 * @since 1.2.0
 */
public class ChinaMapView extends MapView {

  private CoordinateShifter coordinateShifter;

  @UiThread
  public ChinaMapView(@NonNull Context context) {
    super(context);
  }

  @UiThread
  public ChinaMapView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
  }

  @UiThread
  public ChinaMapView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
  }

  @UiThread
  public ChinaMapView(@NonNull Context context, @Nullable MapboxMapOptions options) {
    super(context, options);
  }

  @Override
  protected void initialize(@NonNull Context context, @NonNull MapboxMapOptions mapboxMapOptions) {
    // Initialize coordinate shifter so that all coordinates are shifted to China's
    // GCJ-02 coordinate system.
    enableGeoJsonShifting(new ChinaCoordinateShifter());

    // Adjust the Maps API base URL for China map tile retrieval
    mapboxMapOptions.apiBaseUri(MapboxChinaConstants.BASE_API_URL);

    super.initialize(context, mapboxMapOptions);

    FileSource fileSource = FileSource.getInstance(context);
    fileSource.setResourceTransform(new ChinaStyleMapper());
  }

  static {

  }

  /**
   * Tells the Maps SDK to shift GeoJSON data so that coordinates follow China's GCJ-02 system.
   *
   * @since 1.2.0
   */
  public void enableGeoJsonShifting(CoordinateShifter coordinateShifter) {
    this.coordinateShifter = coordinateShifter;
    setCoordinateShifter(coordinateShifter);
  }

  /**
   * Tells the Maps SDK to not shift any GeoJSON data. This method is particularly helpful if data is already
   * in GCJ-02 format before being fed to the Maps SDK.
   *
   * @since 1.2.0
   */
  public void disableGeoJsonShifting() {
    this.coordinateShifter = null;
    setCoordinateShifter(coordinateShifter);
  }

  /**
   * Retrieve the latest CoordinateShifter interface being used by the Maps SDK.
   *
   * @return the current CoordinateShifter object
   * @since 1.2.0
   */
  public CoordinateShifter getCoordinateShifter() {
    return coordinateShifter;
  }

  /**
   * Pass the set CoordinateShifter interface object to the CoordinateShifterManager class, so that
   * the Maps SDK knows how to handle GeoJSON coordinates.
   *
   * @param coordinateShifter an object of a class which implements the CoordinateShifter interface
   * @since 1.2.0
   */
  private void setCoordinateShifter(@Nullable CoordinateShifter coordinateShifter) {
    CoordinateShifterManager.setCoordinateShifter(coordinateShifter);
  }
}