/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.portal.kernel.nio.intraband;

import com.liferay.petra.executor.PortalExecutorManager;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.util.ServiceProxyFactory;

import java.util.concurrent.Executor;

/**
 * @author Shuyang Zhou
 */
public abstract class BaseAsyncDatagramReceiveHandler
	implements DatagramReceiveHandler {

	public BaseAsyncDatagramReceiveHandler() {
		Class<? extends BaseAsyncDatagramReceiveHandler> clazz = getClass();

		_executor = _portalExecutorManager.getPortalExecutor(clazz.getName());
	}

	@Override
	public void receive(
		RegistrationReference registrationReference, Datagram datagram) {

		_executor.execute(new DispatchJob(registrationReference, datagram));
	}

	protected abstract void doReceive(
			RegistrationReference registrationReference, Datagram datagram)
		throws Exception;

	private static final Log _log = LogFactoryUtil.getLog(
		BaseAsyncDatagramReceiveHandler.class);

	private static volatile PortalExecutorManager _portalExecutorManager =
		ServiceProxyFactory.newServiceTrackedInstance(
			PortalExecutorManager.class, BaseAsyncDatagramReceiveHandler.class,
			"_portalExecutorManager", true);

	private final Executor _executor;

	private class DispatchJob implements Runnable {

		public DispatchJob(
			RegistrationReference registrationReference, Datagram datagram) {

			_registrationReference = registrationReference;
			_datagram = datagram;
		}

		@Override
		public void run() {
			try {
				doReceive(_registrationReference, _datagram);
			}
			catch (Exception exception) {
				_log.error("Unable to dispatch", exception);
			}
		}

		private final Datagram _datagram;
		private final RegistrationReference _registrationReference;

	}

}