/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.lcs.util;

import java.math.BigInteger;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides helper methods for Liferay patches. This class uses standard Liferay
 * patch naming conventions to detect a patch's human name or file name.
 *
 * @author  Ivica Cardic
 * @version 2.1.1
 * @since   LCS 0.1
 */
public class PatchUtil {

	/**
	 * Returns the patch's file name. The <code>patchName</code> is used as a
	 * base to build the patch's file name.
	 *
	 * @param  patchName the patch's name
	 * @return the patch's file name
	 * @since  LCS 0.1
	 */
	public static String getPatchFileName(String patchName) {
		return _LIFERAY_PREFIX.concat(
			patchName
		).concat(
			".zip"
		);
	}

	/**
	 * Returns the patch ID, which is extracted from the patch name.
	 *
	 * @param  patchName the patch's name
	 * @return the patch ID
	 * @since  LCS 0.1
	 */
	public static String getPatchId(String patchName) {
		return patchName.replace(_FIX_PACK_PREFIX, "");
	}

	public static String getPatchMD5Sum(byte[] bytes) {
		try {
			MessageDigest messageDigest = MessageDigest.getInstance("MD5");

			messageDigest.reset();

			messageDigest.update(bytes);

			byte[] digest = messageDigest.digest();

			BigInteger bigInteger = new BigInteger(1, digest);

			return String.format("%032x", bigInteger);
		}
		catch (NoSuchAlgorithmException nsae) {
			_log.error(
				"Skipping MD5 sum check because of missing MD5 sum algorithm");

			return null;
		}
	}

	/**
	 * Returns the patch's name, which is extracted from the
	 * <code>patchFileName</code>.
	 *
	 * @param  patchFileName the patch's file name
	 * @return the patch's name
	 * @since  LCS 0.1
	 */
	public static String getPatchName(String patchFileName) {
		String patchName = patchFileName.replace(_LIFERAY_PREFIX, "");

		return patchName.replace(".zip", "");
	}

	private static final String _FIX_PACK_PREFIX = "fix-pack-";

	private static final String _LIFERAY_PREFIX = "liferay-";

	private static final Logger _log = LoggerFactory.getLogger(PatchUtil.class);

}