/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.content.targeting.util;

import com.liferay.portal.kernel.io.unsync.UnsyncStringWriter;
import com.liferay.portal.kernel.json.JSONArray;
import com.liferay.portal.kernel.json.JSONObject;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.servlet.BufferCacheServletResponse;
import com.liferay.portal.kernel.servlet.taglib.aui.ValidatorTag;
import com.liferay.portal.kernel.template.Template;
import com.liferay.portal.kernel.template.TemplateConstants;
import com.liferay.portal.kernel.template.TemplateManager;
import com.liferay.portal.kernel.template.TemplateManagerUtil;
import com.liferay.portal.kernel.template.URLTemplateResource;
import com.liferay.portal.kernel.util.AggregateResourceBundleLoader;
import com.liferay.portal.kernel.util.ClassResourceBundleLoader;
import com.liferay.portal.kernel.util.ResourceBundleLoader;
import com.liferay.portal.kernel.util.StringPool;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.Validator;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author Julio Camarero
 */
public class ContentTargetingContextUtil {

	public static Map<String, String> getJSONValues(
		JSONArray data, String namespace, String id) {

		Map<String, String> values = new HashMap<>(data.length());

		for (int i = 0; i < data.length(); i++) {
			JSONObject jsonObject = data.getJSONObject(i);

			String name = jsonObject.getString("name");

			name = StringUtil.replace(
				name, new String[] {namespace, id},
				new String[] {StringPool.BLANK, StringPool.BLANK});

			values.put(name, jsonObject.getString("value"));
		}

		return values;
	}

	public static String includeJSP(
			ServletContext servletContext, String path,
			Map<String, Object> context)
		throws Exception {

		if (Validator.isNull(path)) {
			_log.error("Cannot find path " + path);

			return StringPool.BLANK;
		}

		HttpServletRequest request = (HttpServletRequest)context.get("request");
		HttpServletResponse response = (HttpServletResponse)context.get(
			"response");

		request.setAttribute("context", context);

		RequestDispatcher requestDispatcher =
			servletContext.getRequestDispatcher(path);

		BufferCacheServletResponse bufferResponse =
			new BufferCacheServletResponse(response);

		ResourceBundleLoader resourceBundleLoader =
			(ResourceBundleLoader)request.getAttribute(
				WebKeys.RESOURCE_BUNDLE_LOADER);

		try {
			request.setAttribute(
				WebKeys.RESOURCE_BUNDLE_LOADER,
				getResourceBundleLoader(
					servletContext.getClassLoader(),
					(ResourceBundleLoader)context.get(
						"portletResourceBundleLoader")));

			requestDispatcher.include(request, bufferResponse);

			return bufferResponse.getString();
		}
		catch (ServletException se) {
			_log.error("Unable to include " + path, se);
		}
		finally {
			request.setAttribute(
				WebKeys.RESOURCE_BUNDLE_LOADER, resourceBundleLoader);
		}

		return StringPool.BLANK;
	}

	public static String parseTemplate(
			Class clazz, String templatePath, Map<String, Object> context)
		throws Exception {

		if (!templatePath.startsWith("/")) {
			templatePath = "/" + templatePath;
		}

		URLTemplateResource urlTemplateResource = new URLTemplateResource(
			templatePath, clazz.getResource(templatePath));

		Template template = TemplateManagerUtil.getTemplate(
			TemplateConstants.LANG_TYPE_FTL, urlTemplateResource, false);

		ClassLoader resourceBundleClassLoader = (ClassLoader)context.get(
			"resourceBundleClassLoader");

		if (resourceBundleClassLoader == null) {
			resourceBundleClassLoader = clazz.getClassLoader();
		}

		HttpServletRequest request = (HttpServletRequest)context.get("request");
		HttpServletResponse response = (HttpServletResponse)context.get(
			"response");

		// Default template context

		template.prepare(request);

		// Custom template context

		template.putAll(context);

		// Aggregate language bundles

		ResourceBundleLoader resourceBundleLoader =
			(ResourceBundleLoader)request.getAttribute(
				WebKeys.RESOURCE_BUNDLE_LOADER);

		try {
			request.setAttribute(
				WebKeys.RESOURCE_BUNDLE_LOADER,
				getResourceBundleLoader(
					resourceBundleClassLoader,
					(ResourceBundleLoader)context.get(
						"portletResourceBundleLoader")));

			// Taglib support

			TemplateManager templateManager =
				TemplateManagerUtil.getTemplateManager(
					TemplateConstants.LANG_TYPE_FTL);

			templateManager.addTaglibSupport(template, request, response);

			// Render template with custom configuration to support includes

			UnsyncStringWriter unsyncStringWriter = new UnsyncStringWriter();

			template.processTemplate(unsyncStringWriter);

			return unsyncStringWriter.toString();
		}
		finally {
			request.setAttribute(
				WebKeys.RESOURCE_BUNDLE_LOADER, resourceBundleLoader);
		}
	}

	public static String processExceptions(
		List exceptions, Class clazz, Map<String, Object> context) {

		if (Validator.isNull(exceptions)) {
			return StringPool.BLANK;
		}

		try {
			context.put("exceptions", exceptions);
			context.put("resourceBundleClassLoader", clazz.getClassLoader());

			Class<?> portletClass = (Class)context.get("portletClass");

			return parseTemplate(
				portletClass, "templates/ct_exceptions.ftl", context);
		}
		catch (Exception e) {
			_log.error(e);
		}

		return StringPool.BLANK;
	}

	public static String processValidators(
		Map<String, List<ValidatorTag>> validatorTagsMap, Class clazz,
		Map<String, Object> context) {

		if (Validator.isNull(validatorTagsMap)) {
			return StringPool.BLANK;
		}

		try {
			context.put("validatorTagsMap", validatorTagsMap);
			context.put("resourceBundleClassLoader", clazz.getClassLoader());

			Class<?> portletClass = (Class)context.get("portletClass");

			return parseTemplate(
				portletClass, "templates/ct_validators.ftl", context);
		}
		catch (Exception e) {
			_log.error(e);
		}

		return StringPool.BLANK;
	}

	protected static ResourceBundleLoader getResourceBundleLoader(
		ClassLoader classLoader,
		ResourceBundleLoader portletResourceBundleLoader) {

		return new AggregateResourceBundleLoader(
			new ClassResourceBundleLoader("content.Language", classLoader),
			portletResourceBundleLoader);
	}

	private static final Log _log = LogFactoryUtil.getLog(
		ContentTargetingContextUtil.class);

}