/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.content.targeting.api.model;

import com.liferay.content.targeting.anonymous.users.model.AnonymousUser;
import com.liferay.content.targeting.exception.InvalidRuleException;
import com.liferay.content.targeting.model.RuleInstance;
import com.liferay.content.targeting.model.UserSegment;
import com.liferay.exportimport.kernel.lar.PortletDataContext;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.xml.Element;

import java.util.Locale;
import java.util.Map;

import javax.portlet.PortletRequest;
import javax.portlet.PortletResponse;

import javax.servlet.http.HttpServletRequest;

/**
 * Provides the Rule interface, allowing custom rule creation and evaluation for
 * the Content Targeting portlet.
 *
 * @author Eudaldo Alonso
 */
public interface Rule {

	/**
	 * Does processing when the rule is installed.
	 */
	public void activate();

	/**
	 * Does processing when the rule is uninstalled.
	 */
	public void deActivate();

	/**
	 * Removes any additional data added by this rule when the rule instance is
	 * removed.
	 *
	 * @param  ruleInstance the rule instance with the stored configuration
	 * @throws PortalException if a portal exception occurred
	 */
	public void deleteData(RuleInstance ruleInstance) throws PortalException;

	/**
	 * Returns <code>true</code> if the user complies with the rule instance in
	 * real time. The evaluation is completed correctly after the user makes a
	 * request.
	 *
	 * @param  request the servlet request of the user
	 * @param  ruleInstance the rule instance with stored configuration
	 * @param  anonymousUser the user who evaluates the rule
	 * @return <code>true</code> if the user complies with the rule instance in
	 *         real time; <code>false</code> otherwise
	 * @throws Exception if an exception occurred
	 */
	public boolean evaluate(
			HttpServletRequest request, RuleInstance ruleInstance,
			AnonymousUser anonymousUser)
		throws Exception;

	/**
	 * Returns <code>true</code> if the user complies with the rule instance in
	 * an offline mode. The evaluation is completed without having a user
	 * request. This will only be called if the rule supports offline evaluation
	 * (see method {@link #supportsOfflineEvaluation()}). A context map can be
	 * optionally passed with some context variables.
	 *
	 * @param  context a map of objects that can be used to evaluate the rule
	 *         instance
	 * @param  ruleInstance the rule instance with stored configuration
	 * @param  anonymousUser the user who evaluates the rule
	 * @return <code>true</code> if the user complies with the rule instance in
	 *         an offline mode; <code>false</code> otherwise
	 */
	public boolean evaluate(
		Map<String, Object> context, RuleInstance ruleInstance,
		AnonymousUser anonymousUser);

	/**
	 * Exports any additional data added by this rule when the rule instance is
	 * exported.
	 *
	 * @param  portletDataContext the context of the data export
	 * @param  userSegmentElement the element with the user segment export data
	 * @param  userSegment the user segment containing the rule instance
	 * @param  ruleInstanceElement the element with the rule instance export
	 *         data
	 * @param  ruleInstance the rule instance with stored configuration
	 * @throws Exception if an exception occurred
	 */
	public void exportData(
			PortletDataContext portletDataContext, Element userSegmentElement,
			UserSegment userSegment, Element ruleInstanceElement,
			RuleInstance ruleInstance)
		throws Exception;

	/**
	 * Returns the time in milliseconds that the evaluation of this rule can be
	 * cached. For example, an <code>Age</code> rule can be cached at least 1
	 * day and a <code>Geolocation</code> rule could be cached 5 minutes. This
	 * value can be configurable by adding a custom configuration to your
	 * component. A value of <code>0</code> means that the evaluation can not be
	 * cached.
	 *
	 * @return the time in milliseconds
	 */
	public long getCacheTime();

	/**
	 * Returns the rule's localized description.
	 *
	 * @param  locale the language locale
	 * @return the rule's localized description
	 */
	public String getDescription(Locale locale);

	/**
	 * Returns the HTML code containing the form fields required to edit the
	 * rule instance configuration, based on the context.
	 *
	 * @param  ruleInstance the rule instance with stored configuration
	 * @param  context the map defining the form evaluation context
	 * @param  values the values configured by users for the current rule
	 *         instance based on the form controls from the HTML. This will be
	 *         used when there is an error and the form is reloaded.
	 * @return the HTML code containing the form fields required to edit the
	 *         rule instance configuration, based on the context
	 */
	public String getFormHTML(
		RuleInstance ruleInstance, Map<String, Object> context,
		Map<String, String> values);

	/**
	 * Returns the Font Awesome CSS class for the rule icon.
	 *
	 * @return the Font Awesome CSS class for the rule icon
	 * @see    <a href="http://fortawesome.github.io/Font-Awesome/3.2.1/">Font
	 *         Awesome documentation</a>
	 */
	public String getIcon();

	/**
	 * Returns the rule's localized name.
	 *
	 * @param  locale the language locale
	 * @return the rule's localized name
	 */
	public String getName(Locale locale);

	/**
	 * Returns the key that identifies the category of the rule.
	 *
	 * @return the key that identifies the category
	 */
	public String getRuleCategoryKey();

	/**
	 * Returns the key that identifies the rule. The rule instances of this rule
	 * are identified by their rule key.
	 *
	 * @return the key that identifies the rule
	 */
	public String getRuleKey();

	/**
	 * Returns the rule's localized short description.
	 *
	 * @param  locale the language locale
	 * @return the rule's localized short description
	 */
	public String getShortDescription(Locale locale);

	/**
	 * Returns the rule instance's localized summary.
	 *
	 * @param  ruleInstance the rule instance with stored configuration
	 * @param  locale the language locale
	 * @return the rule instance's localized summary
	 */
	public String getSummary(RuleInstance ruleInstance, Locale locale);

	/**
	 * Imports any additional data added by this rule when the rule instance is
	 * imported.
	 *
	 * @param  portletDataContext the context of the data import
	 * @param  userSegment the user segment containing the rule instance
	 * @param  ruleInstance the rule instance with stored configuration
	 * @throws Exception if an exception occurred
	 */
	public void importData(
			PortletDataContext portletDataContext, UserSegment userSegment,
			RuleInstance ruleInstance)
		throws Exception;

	/**
	 * Returns <code>true</code> if the rule can be used more than once with
	 * different values for a user segment.
	 *
	 * @return <code>true</code> if the rule can be used more than once;
	 *         <code>false</code> otherwise
	 */
	public boolean isInstantiable();

	/**
	 * Returns <code>true</code> if the rule is visible.
	 *
	 * @return <code>true</code> if the rule is visible; <code>false</code>
	 *         otherwise
	 */
	public boolean isVisible();

	/**
	 * Returns the result of evaluating the rule form fields in the context of
	 * the request and response.
	 *
	 * @param  portletRequest the request from which to get the request
	 *         parameters
	 * @param  portletResponse the response to receive the render parameters
	 * @param  id the identifier that differentiates between rule instances of
	 *         the same type of an instantiable rule
	 * @param  values the values configured by users for the current rule
	 *         instance based on the form controls from the HTML
	 * @return the result of evaluating the rule form fields in the context of
	 *         the request and response
	 * @throws InvalidRuleException if an invalid rule exception occurred
	 */
	public String processRule(
			PortletRequest portletRequest, PortletResponse portletResponse,
			String id, Map<String, String> values)
		throws InvalidRuleException;

	/**
	 * Returns <code>true</code> if the rule can be evaluated offline (without
	 * the user request in real time). If this is set to <code>true</code>, the
	 * method {@link #evaluate(Map, RuleInstance, AnonymousUser)} should be
	 * implemented.
	 *
	 * @return <code>true</code> if the rule can be evaluated offline;
	 *         <code>false</code> otherwise.
	 */
	public boolean supportsOfflineEvaluation();

}