Liferay.Loader.define("site-navigation-menu-web@1.0.6/js/SiteNavigationMenuItem.es", ['exports', 'metal-dom/src/all/dom'], function (exports, _dom) {
	'use strict';

	Object.defineProperty(exports, "__esModule", {
		value: true
	});
	exports.SiteNavigationMenuItem = exports.MENU_ITEM_DRAGGING_CLASSNAME = exports.MENU_ITEM_DRAG_ICON_CLASSNAME = exports.MENU_ITEM_CLASSNAME = exports.MENU_ITEM_CONTENT_CLASSNAME = exports.MENU_CONTAINER_CLASSNAME = undefined;


	var MENU_CONTAINER_CLASSNAME = 'site-navigation-menu-container';

	var MENU_ITEM_CLASSNAME = 'site-navigation-menu-item';

	var MENU_ITEM_CONTENT_CLASSNAME = MENU_ITEM_CLASSNAME + '__content';

	var MENU_ITEM_DRAGGING_CLASSNAME = MENU_ITEM_CLASSNAME + '--dragging';

	var MENU_ITEM_DRAG_ICON_CLASSNAME = MENU_ITEM_CLASSNAME + '__drag-icon';

	var MENU_ITEM_SELECTED_CLASSNAME = MENU_ITEM_CLASSNAME + '--selected';

	/**
  * Set of static functions for managing site navigation menu item
  * DOM querying and mutation.
  * @type {Object}
  */

	var SiteNavigationMenuItem = {

		/**
   * Returns an array with the menuItem children of the given menuItem.
   * @param {HTMLElement} menuItem
   * @return {Array<HTMLElement>}
   * @review
   */

		getChildren: function getChildren(menuItem) {
			return Array.prototype.slice.call(menuItem.children).filter(function (child) {
				return (0, _dom.hasClass)(child, MENU_ITEM_CLASSNAME);
			});
		},

		/**
   * Returns a menuItem element, parent of a given menuItemContent.
   * @param {HTMLElement} menuItemContent
   * @return {HTMLElement|null}
   * @review
   */

		getFromContentElement: function getFromContentElement(menuItemContent) {
			return (0, _dom.closest)(menuItemContent, '.' + MENU_ITEM_CLASSNAME);
		},

		/**
   * Returns a menuItem element with the given ID
   * @param {number|string} menuItemId
   * @return {HTMLElement|null}
   */

		getFromId: function getFromId(menuItemId) {
			return document.querySelector('.' + MENU_ITEM_CLASSNAME + '[data-site-navigation-menu-item-id="' + menuItemId + '"]');
		},

		/**
   * Gets the ID of a given menuItem element.
   * @param {HTMLElement} menuItem
   * @return {number}
   */

		getId: function getId(menuItem) {
			return parseInt(menuItem.dataset.siteNavigationMenuItemId, 10) || 0;
		},

		getNextSibling: function getNextSibling(menuItem) {
			(0, _dom.next)(menuItem, '.' + MENU_ITEM_CLASSNAME);
		},


		/**
   * Returns a menuItem element, parent of a given menuItem.
   * @param {HTMLElement} menuItem
   * @return {HTMLElement|null}
   * @review
   */

		getParent: function getParent(menuItem) {
			return menuItem.parentElement;
		},

		/**
   * For a given menuItem element, returns it's sibblings
   * @param {HTMLElement} menuItem
   * @return {Array<HTMLElement>}
   * @review
   */

		getSiblings: function getSiblings(menuItem) {
			var parentElement = menuItem.parentElement;
			var siblings = [];

			if (parentElement) {
				siblings = SiteNavigationMenuItem.getChildren(parentElement);
			}

			return siblings;
		},

		/**
   * Returns true if the given menuItem is child of the given parentMenuItem
   * @param {HTMLElement} menuItem
   * @param {HTMLElement} parentMenuItem
   * @return {boolean}
   */

		isChildOf: function isChildOf(menuItem, parentMenuItem) {
			return (0, _dom.contains)(parentMenuItem, menuItem);
		},

		/**
   * Returns true if the given htmlElement is a menuItem, false otherwise.
   * @param {HTMLElement} htmlElement
   * @return {boolean}
   */

		isMenuItem: function isMenuItem(htmlElement) {
			return (0, _dom.hasClass)(htmlElement, MENU_ITEM_CLASSNAME);
		},

		/**
   * Returns true if the given menuItem element is selected, false otherwise
   * @param {HTMLElement} menuItem
   * @return {boolean}
   */

		isSelected: function isSelected(menuItem) {
			return (0, _dom.hasClass)(menuItem, MENU_ITEM_SELECTED_CLASSNAME);
		},

		/**
   * Mutates the given menuItem element by changing it's status
   * to dragging/not dragging.
   * @param {HTMLElement} menuItem
   */

		setDragging: function setDragging(menuItem) {
			var dragging = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

			if (dragging) {
				(0, _dom.addClasses)(menuItem, MENU_ITEM_DRAGGING_CLASSNAME);
			} else {
				(0, _dom.removeClasses)(menuItem, MENU_ITEM_DRAGGING_CLASSNAME);
			}
		},

		/**
   * Mutates the given menuItem by changing it's status
   * to selected/unselected. Only a single menuItem can be
   * selected, so if selected is changed to true,
   * any other selected menuItem will be unselected.
   * @param {HTMLElement} menuItem
   * @param {boolean} [selected=false]
   */

		setSelected: function setSelected(menuItem) {
			var selected = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

			if (selected) {
				var selectedMenuItem = (0, _dom.toElement)('.' + MENU_ITEM_SELECTED_CLASSNAME);

				if (selectedMenuItem) {
					SiteNavigationMenuItem.setSelected(selectedMenuItem, false);
				}

				(0, _dom.addClasses)(menuItem, MENU_ITEM_SELECTED_CLASSNAME);
			} else {
				(0, _dom.removeClasses)(menuItem, MENU_ITEM_SELECTED_CLASSNAME);
			}
		}
	};

	exports.MENU_CONTAINER_CLASSNAME = MENU_CONTAINER_CLASSNAME;
	exports.MENU_ITEM_CONTENT_CLASSNAME = MENU_ITEM_CONTENT_CLASSNAME;
	exports.MENU_ITEM_CLASSNAME = MENU_ITEM_CLASSNAME;
	exports.MENU_ITEM_DRAG_ICON_CLASSNAME = MENU_ITEM_DRAG_ICON_CLASSNAME;
	exports.MENU_ITEM_DRAGGING_CLASSNAME = MENU_ITEM_DRAGGING_CLASSNAME;
	exports.SiteNavigationMenuItem = SiteNavigationMenuItem;
	exports.default = SiteNavigationMenuItem;
});
//# sourceMappingURL=SiteNavigationMenuItem.es.js.map
