"use strict";

Liferay.Loader.define("segments-web$dnd-core@7.7.0/lib/cjs/HandlerRegistryImpl", ['module', 'exports', 'require', './actions/registry', './utils/getNextUniqueId', './interfaces', './contracts', 'segments-web$invariant', 'segments-web$asap'], function (module, exports, require) {
    var define = undefined;
    Object.defineProperty(exports, "__esModule", { value: true });
    var registry_1 = require("./actions/registry");
    var getNextUniqueId_1 = require("./utils/getNextUniqueId");
    var interfaces_1 = require("./interfaces");
    var contracts_1 = require("./contracts");
    var invariant = require("segments-web$invariant");
    var asap = require("segments-web$asap");
    function getNextHandlerId(role) {
        var id = getNextUniqueId_1.default().toString();
        switch (role) {
            case interfaces_1.HandlerRole.SOURCE:
                return "S" + id;
            case interfaces_1.HandlerRole.TARGET:
                return "T" + id;
            default:
                throw new Error("Unknown Handler Role: " + role);
        }
    }
    function parseRoleFromHandlerId(handlerId) {
        switch (handlerId[0]) {
            case 'S':
                return interfaces_1.HandlerRole.SOURCE;
            case 'T':
                return interfaces_1.HandlerRole.TARGET;
            default:
                invariant(false, "Cannot parse handler ID: " + handlerId);
        }
    }
    function mapContainsValue(map, searchValue) {
        var entries = map.entries();
        var isDone = false;
        do {
            var _a = entries.next(),
                done = _a.done,
                _b = _a.value,
                value = _b[1];
            if (value === searchValue) {
                return true;
            }
            isDone = done;
        } while (!isDone);
        return false;
    }
    var HandlerRegistryImpl = /** @class */function () {
        function HandlerRegistryImpl(store) {
            this.store = store;
            this.types = new Map();
            this.dragSources = new Map();
            this.dropTargets = new Map();
            this.pinnedSourceId = null;
            this.pinnedSource = null;
        }
        HandlerRegistryImpl.prototype.addSource = function (type, source) {
            contracts_1.validateType(type);
            contracts_1.validateSourceContract(source);
            var sourceId = this.addHandler(interfaces_1.HandlerRole.SOURCE, type, source);
            this.store.dispatch(registry_1.addSource(sourceId));
            return sourceId;
        };
        HandlerRegistryImpl.prototype.addTarget = function (type, target) {
            contracts_1.validateType(type, true);
            contracts_1.validateTargetContract(target);
            var targetId = this.addHandler(interfaces_1.HandlerRole.TARGET, type, target);
            this.store.dispatch(registry_1.addTarget(targetId));
            return targetId;
        };
        HandlerRegistryImpl.prototype.containsHandler = function (handler) {
            return mapContainsValue(this.dragSources, handler) || mapContainsValue(this.dropTargets, handler);
        };
        HandlerRegistryImpl.prototype.getSource = function (sourceId, includePinned) {
            if (includePinned === void 0) {
                includePinned = false;
            }
            invariant(this.isSourceId(sourceId), 'Expected a valid source ID.');
            var isPinned = includePinned && sourceId === this.pinnedSourceId;
            var source = isPinned ? this.pinnedSource : this.dragSources.get(sourceId);
            return source;
        };
        HandlerRegistryImpl.prototype.getTarget = function (targetId) {
            invariant(this.isTargetId(targetId), 'Expected a valid target ID.');
            return this.dropTargets.get(targetId);
        };
        HandlerRegistryImpl.prototype.getSourceType = function (sourceId) {
            invariant(this.isSourceId(sourceId), 'Expected a valid source ID.');
            return this.types.get(sourceId);
        };
        HandlerRegistryImpl.prototype.getTargetType = function (targetId) {
            invariant(this.isTargetId(targetId), 'Expected a valid target ID.');
            return this.types.get(targetId);
        };
        HandlerRegistryImpl.prototype.isSourceId = function (handlerId) {
            var role = parseRoleFromHandlerId(handlerId);
            return role === interfaces_1.HandlerRole.SOURCE;
        };
        HandlerRegistryImpl.prototype.isTargetId = function (handlerId) {
            var role = parseRoleFromHandlerId(handlerId);
            return role === interfaces_1.HandlerRole.TARGET;
        };
        HandlerRegistryImpl.prototype.removeSource = function (sourceId) {
            var _this = this;
            invariant(this.getSource(sourceId), 'Expected an existing source.');
            this.store.dispatch(registry_1.removeSource(sourceId));
            asap(function () {
                _this.dragSources.delete(sourceId);
                _this.types.delete(sourceId);
            });
        };
        HandlerRegistryImpl.prototype.removeTarget = function (targetId) {
            invariant(this.getTarget(targetId), 'Expected an existing target.');
            this.store.dispatch(registry_1.removeTarget(targetId));
            this.dropTargets.delete(targetId);
            this.types.delete(targetId);
        };
        HandlerRegistryImpl.prototype.pinSource = function (sourceId) {
            var source = this.getSource(sourceId);
            invariant(source, 'Expected an existing source.');
            this.pinnedSourceId = sourceId;
            this.pinnedSource = source;
        };
        HandlerRegistryImpl.prototype.unpinSource = function () {
            invariant(this.pinnedSource, 'No source is pinned at the time.');
            this.pinnedSourceId = null;
            this.pinnedSource = null;
        };
        HandlerRegistryImpl.prototype.addHandler = function (role, type, handler) {
            var id = getNextHandlerId(role);
            this.types.set(id, type);
            if (role === interfaces_1.HandlerRole.SOURCE) {
                this.dragSources.set(id, handler);
            } else if (role === interfaces_1.HandlerRole.TARGET) {
                this.dropTargets.set(id, handler);
            }
            return id;
        };
        return HandlerRegistryImpl;
    }();
    exports.default = HandlerRegistryImpl;
});
//# sourceMappingURL=HandlerRegistryImpl.js.map