/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.portal.settings.authentication.ldap.web.internal.portlet.action;

import com.liferay.portal.kernel.portlet.bridges.mvc.BaseFormMVCActionCommand;
import com.liferay.portal.kernel.portlet.bridges.mvc.MVCActionCommand;
import com.liferay.portal.kernel.security.auth.PrincipalException;
import com.liferay.portal.kernel.security.permission.PermissionChecker;
import com.liferay.portal.kernel.security.permission.PermissionThreadLocal;
import com.liferay.portal.kernel.servlet.SessionErrors;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.ParamUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.kernel.util.WebKeys;
import com.liferay.portal.security.ldap.authenticator.configuration.LDAPAuthConfiguration;
import com.liferay.portal.security.ldap.configuration.ConfigurationProvider;
import com.liferay.portal.security.ldap.configuration.LDAPServerConfiguration;
import com.liferay.portal.security.ldap.constants.LDAPConstants;
import com.liferay.portal.security.ldap.exportimport.configuration.LDAPExportConfiguration;
import com.liferay.portal.security.ldap.exportimport.configuration.LDAPImportConfiguration;
import com.liferay.portal.settings.constants.PortalSettingsPortletKeys;

import java.util.Dictionary;
import java.util.List;
import java.util.stream.Stream;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Tomas Polesovsky
 */
@Component(
	immediate = true,
	property = {
		"javax.portlet.name=" + PortalSettingsPortletKeys.PORTAL_SETTINGS,
		"mvc.command.name=/portal_settings/ldap"
	},
	service = MVCActionCommand.class
)
public class PortalSettingsLDAPFormMVCActionCommand
	extends BaseFormMVCActionCommand {

	@Override
	protected void doProcessAction(
			ActionRequest actionRequest, ActionResponse actionResponse)
		throws Exception {

		ThemeDisplay themeDisplay = (ThemeDisplay)actionRequest.getAttribute(
			WebKeys.THEME_DISPLAY);

		PermissionChecker permissionChecker =
			PermissionThreadLocal.getPermissionChecker();

		if (!permissionChecker.isCompanyAdmin(themeDisplay.getCompanyId())) {
			SessionErrors.add(actionRequest, PrincipalException.class);

			actionResponse.setRenderParameter("mvcPath", "/error.jsp");

			return;
		}

		updateBooleanProperties(
			actionRequest, _ldapAuthConfigurationProvider,
			themeDisplay.getCompanyId(), LDAPConstants.AUTH_ENABLED,
			LDAPConstants.AUTH_REQUIRED, LDAPConstants.PASSWORD_POLICY_ENABLED);

		updateStringProperties(
			actionRequest, _ldapAuthConfigurationProvider,
			themeDisplay.getCompanyId(), LDAPConstants.AUTH_METHOD,
			LDAPConstants.PASSWORD_ENCRYPTION_ALGORITHM);

		updateBooleanProperties(
			actionRequest, _ldapExportConfigurationProvider,
			themeDisplay.getCompanyId(), LDAPConstants.EXPORT_ENABLED,
			LDAPConstants.EXPORT_GROUP_ENABLED);

		updateBooleanProperties(
			actionRequest, _ldapImportConfigurationProvider,
			themeDisplay.getCompanyId(),
			LDAPConstants.IMPORT_CREATE_ROLE_PER_GROUP,
			LDAPConstants.IMPORT_ENABLED,
			LDAPConstants.IMPORT_GROUP_CACHE_ENABLED,
			LDAPConstants.IMPORT_ON_STARTUP,
			LDAPConstants.IMPORT_USER_PASSWORD_AUTOGENERATED,
			LDAPConstants.IMPORT_USER_PASSWORD_DEFAULT,
			LDAPConstants.IMPORT_USER_PASSWORD_ENABLED);

		updateIntegerProperties(
			actionRequest, _ldapImportConfigurationProvider,
			themeDisplay.getCompanyId(), LDAPConstants.IMPORT_INTERVAL);

		updateLongProperties(
			actionRequest, _ldapImportConfigurationProvider,
			themeDisplay.getCompanyId(),
			LDAPConstants.IMPORT_LOCK_EXPIRATION_TIME);

		updateStringProperties(
			actionRequest, _ldapImportConfigurationProvider,
			themeDisplay.getCompanyId(), LDAPConstants.IMPORT_METHOD,
			LDAPConstants.IMPORT_USER_PASSWORD_DEFAULT,
			LDAPConstants.IMPORT_USER_SYNC_STRATEGY);

		sortLdapServerConfigurations(
			themeDisplay.getCompanyId(),
			ParamUtil.getString(
				actionRequest,
				"ldap--" + LDAPConstants.AUTH_SERVER_PRIORITY + "--"));
	}

	@Override
	protected void doValidateForm(
		ActionRequest actionRequest, ActionResponse actionResponse) {

		ThemeDisplay themeDisplay = (ThemeDisplay)actionRequest.getAttribute(
			WebKeys.THEME_DISPLAY);

		LDAPImportConfiguration ldapImportConfiguration =
			_ldapImportConfigurationProvider.getConfiguration(
				themeDisplay.getCompanyId());

		if (!ldapImportConfiguration.importUserPasswordAutogenerated()) {
			return;
		}

		boolean ldapExportEnabled = ParamUtil.getBoolean(
			actionRequest, "ldap--" + LDAPConstants.EXPORT_ENABLED + "--");
		boolean ldapImportEnabled = ParamUtil.getBoolean(
			actionRequest, "ldap--" + LDAPConstants.IMPORT_ENABLED + "--");

		if (ldapExportEnabled && ldapImportEnabled) {
			SessionErrors.add(
				actionRequest, "ldapExportAndImportOnPasswordAutogeneration");
		}
	}

	@Reference(
		target = "(factoryPid=com.liferay.portal.security.ldap.authenticator.configuration.LDAPAuthConfiguration)",
		unbind = "-"
	)
	protected void setLDAPAuthConfigurationProvider(
		ConfigurationProvider<LDAPAuthConfiguration>
			ldapAuthConfigurationProvider) {

		_ldapAuthConfigurationProvider = ldapAuthConfigurationProvider;
	}

	@Reference(
		target = "(factoryPid=com.liferay.portal.security.ldap.exportimport.configuration.LDAPExportConfiguration)",
		unbind = "-"
	)
	protected void setLDAPExportConfigurationProvider(
		ConfigurationProvider<LDAPExportConfiguration>
			ldapExportConfigurationProvider) {

		_ldapExportConfigurationProvider = ldapExportConfigurationProvider;
	}

	@Reference(
		target = "(factoryPid=com.liferay.portal.security.ldap.exportimport.configuration.LDAPImportConfiguration)",
		unbind = "-"
	)
	protected void setLDAPImportConfigurationProvider(
		ConfigurationProvider<LDAPImportConfiguration>
			ldapImportConfigurationProvider) {

		_ldapImportConfigurationProvider = ldapImportConfigurationProvider;
	}

	@Reference(
		target = "(factoryPid=com.liferay.portal.security.ldap.configuration.LDAPServerConfiguration)",
		unbind = "-"
	)
	protected void setLDAPServerConfigurationProvider(
		ConfigurationProvider<LDAPServerConfiguration>
			ldapServerConfigurationProvider) {

		_ldapServerConfigurationProvider = ldapServerConfigurationProvider;
	}

	protected void sortLdapServerConfigurations(
		long companyId, String orderedLdapServerIdsString) {

		if (Validator.isBlank(orderedLdapServerIdsString)) {
			return;
		}

		String[] orderedLdapServerIds = orderedLdapServerIdsString.split(",");

		List<Dictionary<String, Object>> dictionaries =
			_ldapServerConfigurationProvider.getConfigurationsProperties(
				companyId);

		for (int i = 0; i < orderedLdapServerIds.length; i++) {
			final int authServerPriority = i;
			long ldapServerId = GetterUtil.getLong(orderedLdapServerIds[i]);

			Stream<Dictionary<String, Object>> stream = dictionaries.stream();

			stream.filter(
				dictionary -> GetterUtil.getLong(
					dictionary.get(LDAPConstants.LDAP_SERVER_ID)) ==
						ldapServerId
			).findFirst(
			).ifPresent(
				dictionary -> {
					dictionary.put(
						LDAPConstants.AUTH_SERVER_PRIORITY, authServerPriority);

					_ldapServerConfigurationProvider.updateProperties(
						companyId,
						GetterUtil.getLong(
							dictionary.get(LDAPConstants.LDAP_SERVER_ID)),
						dictionary);
				}
			);
		}
	}

	protected void updateBooleanProperties(
		ActionRequest actionRequest,
		ConfigurationProvider<?> configurationProvider, long companyId,
		String... propertyNames) {

		Dictionary<String, Object> properties =
			configurationProvider.getConfigurationProperties(companyId);

		for (String propertyName : propertyNames) {
			boolean value = ParamUtil.getBoolean(
				actionRequest, "ldap--" + propertyName + "--");

			properties.put(propertyName, value);
		}

		configurationProvider.updateProperties(companyId, properties);
	}

	protected void updateIntegerProperties(
		ActionRequest actionRequest,
		ConfigurationProvider<?> configurationProvider, long companyId,
		String... propertyNames) {

		Dictionary<String, Object> properties =
			configurationProvider.getConfigurationProperties(companyId);

		for (String propertyName : propertyNames) {
			int value = ParamUtil.getInteger(
				actionRequest, "ldap--" + propertyName + "--");

			properties.put(propertyName, value);
		}

		configurationProvider.updateProperties(companyId, properties);
	}

	protected void updateLongProperties(
		ActionRequest actionRequest,
		ConfigurationProvider<?> configurationProvider, long companyId,
		String... propertyNames) {

		Dictionary<String, Object> properties =
			configurationProvider.getConfigurationProperties(companyId);

		for (String propertyName : propertyNames) {
			long value = ParamUtil.getLong(
				actionRequest, "ldap--" + propertyName + "--");

			properties.put(propertyName, value);
		}

		configurationProvider.updateProperties(companyId, properties);
	}

	protected void updateStringProperties(
		ActionRequest actionRequest,
		ConfigurationProvider<?> configurationProvider, long companyId,
		String... propertyNames) {

		Dictionary<String, Object> properties =
			configurationProvider.getConfigurationProperties(companyId);

		for (String propertyName : propertyNames) {
			String value = ParamUtil.getString(
				actionRequest, "ldap--" + propertyName + "--");

			properties.put(propertyName, value);
		}

		configurationProvider.updateProperties(companyId, properties);
	}

	private ConfigurationProvider<LDAPAuthConfiguration>
		_ldapAuthConfigurationProvider;
	private ConfigurationProvider<LDAPExportConfiguration>
		_ldapExportConfigurationProvider;
	private ConfigurationProvider<LDAPImportConfiguration>
		_ldapImportConfigurationProvider;
	private ConfigurationProvider<LDAPServerConfiguration>
		_ldapServerConfigurationProvider;

}