/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
package org.elasticsearch.xpack.core.watcher.transport.actions.stats;

import org.elasticsearch.Version;
import org.elasticsearch.action.ActionRequestValidationException;
import org.elasticsearch.action.support.nodes.BaseNodeRequest;
import org.elasticsearch.action.support.nodes.BaseNodesRequest;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;

import java.io.IOException;

/**
 * The Request to get the watcher stats
 */
public class WatcherStatsRequest extends BaseNodesRequest<WatcherStatsRequest> {

    private boolean includeCurrentWatches;
    private boolean includeQueuedWatches;
    private boolean includeStats;

    public WatcherStatsRequest() {
    }

    public boolean includeCurrentWatches() {
        return includeCurrentWatches;
    }

    public void includeCurrentWatches(boolean currentWatches) {
        this.includeCurrentWatches = currentWatches;
    }

    public boolean includeQueuedWatches() {
        return includeQueuedWatches;
    }

    public void includeQueuedWatches(boolean includeQueuedWatches) {
        this.includeQueuedWatches = includeQueuedWatches;
    }

    public boolean includeStats() {
        return includeStats;
    }

    public void includeStats(boolean includeStats) {
        this.includeStats = includeStats;
    }

    @Override
    public ActionRequestValidationException validate() {
        return null;
    }

    @Override
    public void readFrom(StreamInput in) throws IOException {
        super.readFrom(in);
        includeCurrentWatches = in.readBoolean();
        includeQueuedWatches = in.readBoolean();
        if (in.getVersion().onOrAfter(Version.V_6_3_0)) {
            includeStats = in.readBoolean();
        } else {
            includeStats = false;
        }
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        super.writeTo(out);
        out.writeBoolean(includeCurrentWatches);
        out.writeBoolean(includeQueuedWatches);
        if (out.getVersion().onOrAfter(Version.V_6_3_0)) {
            out.writeBoolean(includeStats);
        }
    }

    @Override
    public String toString() {
        return "watcher_stats";
    }

    public static class Node extends BaseNodeRequest {

        private boolean includeCurrentWatches;
        private boolean includeQueuedWatches;
        private boolean includeStats;

        public Node() {}

        public Node(WatcherStatsRequest request, String nodeId) {
            super(nodeId);
            includeCurrentWatches = request.includeCurrentWatches();
            includeQueuedWatches = request.includeQueuedWatches();

            includeStats = request.includeStats();
        }

        public boolean includeCurrentWatches() {
            return includeCurrentWatches;
        }

        public boolean includeQueuedWatches() {
            return includeQueuedWatches;
        }

        public boolean includeStats() {
            return includeStats;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            includeCurrentWatches = in.readBoolean();
            includeQueuedWatches = in.readBoolean();
            if (in.getVersion().onOrAfter(Version.V_6_3_0)) {
                includeStats = in.readBoolean();
            } else {
                includeStats = false;
            }
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeBoolean(includeCurrentWatches);
            out.writeBoolean(includeQueuedWatches);
            if (out.getVersion().onOrAfter(Version.V_6_3_0)) {
                out.writeBoolean(includeStats);
            }
        }
    }
}