/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
package org.elasticsearch.xpack.core.security.user;

import org.elasticsearch.Version;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;

import java.io.IOException;

public class InternalUserSerializationHelper {
    public static User readFrom(StreamInput input) throws IOException {
        final boolean isInternalUser = input.readBoolean();
        final String username = input.readString();
        if (isInternalUser) {
            if (SystemUser.is(username)) {
                return SystemUser.INSTANCE;
            } else if (XPackUser.is(username)) {
                return XPackUser.INSTANCE;
            } else if (XPackSecurityUser.is(username)) {
                return XPackSecurityUser.INSTANCE;
            }
            throw new IllegalStateException("user [" + username + "] is not an internal user");
        }
        return User.partialReadFrom(username, input);
    }
    public static void writeTo(User user, StreamOutput output) throws IOException {

        if (SystemUser.is(user)) {
            output.writeBoolean(true);
            output.writeString(SystemUser.NAME);
        } else if (XPackUser.is(user)) {
            output.writeBoolean(true);
            output.writeString(XPackUser.NAME);
        } else if (XPackSecurityUser.is(user) && output.getVersion().onOrAfter(Version.V_5_6_1)) {
            output.writeBoolean(true);
            output.writeString(XPackSecurityUser.NAME);

        } else {
            User.writeTo(user, output);
        }
    }
}
