/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.oauth2.provider.web.internal.portlet.action;

import com.liferay.document.library.kernel.service.DLAppService;
import com.liferay.document.library.util.DLURLHelper;
import com.liferay.oauth2.provider.configuration.OAuth2ProviderConfiguration;
import com.liferay.oauth2.provider.constants.ClientProfile;
import com.liferay.oauth2.provider.constants.GrantType;
import com.liferay.oauth2.provider.model.OAuth2Application;
import com.liferay.oauth2.provider.service.OAuth2ApplicationScopeAliasesLocalService;
import com.liferay.oauth2.provider.service.OAuth2ApplicationService;
import com.liferay.oauth2.provider.util.OAuth2SecureRandomGenerator;
import com.liferay.oauth2.provider.web.internal.constants.OAuth2ProviderPortletKeys;
import com.liferay.oauth2.provider.web.internal.display.context.OAuth2AdminPortletDisplayContext;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.configuration.metatype.bnd.util.ConfigurableUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.portlet.bridges.mvc.MVCActionCommand;
import com.liferay.portal.kernel.repository.model.FileEntry;
import com.liferay.portal.kernel.service.ServiceContext;
import com.liferay.portal.kernel.service.ServiceContextFactory;
import com.liferay.portal.kernel.servlet.SessionErrors;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.ParamUtil;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.kernel.util.WebKeys;

import java.io.InputStream;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletPreferences;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Tomas Polesovsky
 * @author Stian Sigvartsen
 */
@Component(
	configurationPid = "com.liferay.oauth2.provider.configuration.OAuth2ProviderConfiguration",
	property = {
		"javax.portlet.name=" + OAuth2ProviderPortletKeys.OAUTH2_ADMIN,
		"mvc.command.name=/admin/update_oauth2_application"
	},
	service = MVCActionCommand.class
)
public class UpdateOAuth2ApplicationMVCActionCommand
	implements MVCActionCommand {

	@Override
	public boolean processAction(
		ActionRequest request, ActionResponse response) {

		ThemeDisplay themeDisplay = (ThemeDisplay)request.getAttribute(
			WebKeys.THEME_DISPLAY);

		long oAuth2ApplicationId = ParamUtil.getLong(
			request, "oAuth2ApplicationId");

		int clientProfileId = ParamUtil.getInteger(request, "clientProfile");

		ClientProfile clientProfile = getClientProfile(clientProfileId);

		PortletPreferences portletPreferences = request.getPreferences();

		OAuth2AdminPortletDisplayContext oAuth2AdminPortletDisplayContext =
			new OAuth2AdminPortletDisplayContext(
				_oAuth2ApplicationService,
				_oAuth2ApplicationScopeAliasesLocalService,
				_oAuth2ProviderConfiguration, request, null, _dlurlHelper);

		String[] oAuth2Features =
			oAuth2AdminPortletDisplayContext.getOAuth2Features(
				portletPreferences);

		List<String> featuresList = new ArrayList<>();

		for (String feature : oAuth2Features) {
			if (ParamUtil.getBoolean(request, "feature-" + feature, false)) {
				featuresList.add(feature);
			}
		}

		List<GrantType> allowedGrantTypesList = new ArrayList<>();
		List<GrantType> grantTypes =
			oAuth2AdminPortletDisplayContext.getGrantTypes(portletPreferences);

		for (GrantType grantType : clientProfile.grantTypes()) {
			if (!grantTypes.contains(grantType)) {
				continue;
			}

			if (ParamUtil.getBoolean(request, "grant-" + grantType.name())) {
				allowedGrantTypesList.add(grantType);
			}
		}

		String clientId = ParamUtil.get(request, "clientId", StringPool.BLANK);
		String clientSecret = ParamUtil.get(
			request, "clientSecret", StringPool.BLANK);
		String description = ParamUtil.get(
			request, "description", StringPool.BLANK);
		String homePageURL = ParamUtil.get(
			request, "homePageURL", StringPool.BLANK);
		String name = ParamUtil.get(request, "name", StringPool.BLANK);
		String privacyPolicyURL = ParamUtil.get(
			request, "privacyPolicyURL", StringPool.BLANK);
		List<String> redirectURIsList = Arrays.asList(
			StringUtil.splitLines(
				ParamUtil.get(request, "redirectURIs", StringPool.BLANK)));
		List<String> scopeAliasesList = Collections.emptyList();
		long clientCredentialUserId = ParamUtil.get(
			request, "clientCredentialUserId", themeDisplay.getUserId());

		try {
			ServiceContext serviceContext = ServiceContextFactory.getInstance(
				OAuth2Application.class.getName(), request);

			if (oAuth2ApplicationId == 0) {
				if (Validator.isBlank(clientId)) {
					clientId = OAuth2SecureRandomGenerator.generateClientId();
				}

				for (GrantType grantType : allowedGrantTypesList) {
					if (!grantType.isSupportsPublicClients()) {
						clientSecret =
							OAuth2SecureRandomGenerator.generateClientSecret();
					}
				}

				OAuth2Application oAuth2Application =
					_oAuth2ApplicationService.addOAuth2Application(
						allowedGrantTypesList, clientCredentialUserId, clientId,
						clientProfile.id(), clientSecret, description,
						featuresList, homePageURL, 0, name, privacyPolicyURL,
						redirectURIsList, scopeAliasesList, serviceContext);

				response.setRenderParameter(
					"oAuth2ApplicationId",
					String.valueOf(oAuth2Application.getOAuth2ApplicationId()));
			}
			else {
				OAuth2Application oAuth2Application =
					_oAuth2ApplicationService.getOAuth2Application(
						oAuth2ApplicationId);

				_oAuth2ApplicationService.updateOAuth2Application(
					oAuth2ApplicationId, allowedGrantTypesList,
					clientCredentialUserId, clientId, clientProfile.id(),
					clientSecret, description, featuresList, homePageURL,
					oAuth2Application.getIconFileEntryId(), name,
					privacyPolicyURL, redirectURIsList,
					oAuth2Application.getOAuth2ApplicationScopeAliasesId(),
					serviceContext);

				long fileEntryId = ParamUtil.getLong(request, "fileEntryId");

				if (ParamUtil.getBoolean(request, "deleteLogo")) {
					_oAuth2ApplicationService.updateIcon(
						oAuth2ApplicationId, null);
				}
				else if (fileEntryId > 0) {
					FileEntry fileEntry = _dlAppService.getFileEntry(
						fileEntryId);

					InputStream inputStream = fileEntry.getContentStream();

					_oAuth2ApplicationService.updateIcon(
						oAuth2ApplicationId, inputStream);

					_dlAppService.deleteFileEntry(fileEntryId);
				}
			}
		}
		catch (PortalException pe) {
			if (_log.isDebugEnabled()) {
				_log.debug(pe, pe);
			}

			Class<?> peClass = pe.getClass();

			SessionErrors.add(request, peClass.getName(), pe);
		}

		String backURL = ParamUtil.get(request, "backURL", StringPool.BLANK);

		response.setRenderParameter("redirect", backURL);

		return true;
	}

	@Activate
	protected void activate(Map<String, Object> properties) {
		_oAuth2ProviderConfiguration = ConfigurableUtil.createConfigurable(
			OAuth2ProviderConfiguration.class, properties);
	}

	protected ClientProfile getClientProfile(int clientProfileId) {
		for (ClientProfile clientProfile : ClientProfile.values()) {
			if (clientProfile.id() == clientProfileId) {
				return clientProfile;
			}
		}

		throw new IllegalArgumentException(
			"No client profile found for " + clientProfileId);
	}

	private static final Log _log = LogFactoryUtil.getLog(
		UpdateOAuth2ApplicationMVCActionCommand.class);

	@Reference
	private DLAppService _dlAppService;

	@Reference
	private DLURLHelper _dlurlHelper;

	@Reference
	private OAuth2ApplicationScopeAliasesLocalService
		_oAuth2ApplicationScopeAliasesLocalService;

	@Reference
	private OAuth2ApplicationService _oAuth2ApplicationService;

	private OAuth2ProviderConfiguration _oAuth2ProviderConfiguration;

}