/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.knowledge.base.service.permission;

import com.liferay.knowledge.base.model.KBTemplate;
import com.liferay.knowledge.base.service.KBTemplateLocalServiceUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.security.auth.PrincipalException;
import com.liferay.portal.kernel.security.permission.BaseModelPermissionChecker;
import com.liferay.portal.kernel.security.permission.PermissionChecker;

import org.osgi.service.component.annotations.Component;

/**
 * @author Peter Shin
 * @author Brian Wing Shun Chan
 * @author Roberto Díaz
 */
@Component(
	property = {"model.class.name=com.liferay.knowledge.base.model.KBTemplate"},
	service = BaseModelPermissionChecker.class
)
public class KBTemplatePermission implements BaseModelPermissionChecker {

	public static void check(
			PermissionChecker permissionChecker, KBTemplate kbTemplate,
			String actionId)
		throws PortalException {

		if (!contains(permissionChecker, kbTemplate, actionId)) {
			throw new PrincipalException();
		}
	}

	public static void check(
			PermissionChecker permissionChecker, long kbTemplateId,
			String actionId)
		throws PortalException {

		if (!contains(permissionChecker, kbTemplateId, actionId)) {
			throw new PrincipalException();
		}
	}

	public static boolean contains(
		PermissionChecker permissionChecker, KBTemplate kbTemplate,
		String actionId) {

		if (permissionChecker.hasOwnerPermission(
				kbTemplate.getCompanyId(), KBTemplate.class.getName(),
				kbTemplate.getKbTemplateId(), kbTemplate.getUserId(),
				actionId)) {

			return true;
		}

		return permissionChecker.hasPermission(
			kbTemplate.getGroupId(), KBTemplate.class.getName(),
			kbTemplate.getKbTemplateId(), actionId);
	}

	public static boolean contains(
			PermissionChecker permissionChecker, long kbTemplateId,
			String actionId)
		throws PortalException {

		KBTemplate kbTemplate = KBTemplateLocalServiceUtil.getKBTemplate(
			kbTemplateId);

		return contains(permissionChecker, kbTemplate, actionId);
	}

	@Override
	public void checkBaseModel(
			PermissionChecker permissionChecker, long groupId, long primaryKey,
			String actionId)
		throws PortalException {

		check(permissionChecker, primaryKey, actionId);
	}

}