'use strict';

Liferay.Loader.define('frontend-js-metal-web$metal-incremental-dom@2.16.5/src/render/patch', ['module', 'exports', 'require'], function (module, exports, require) {
	var define = undefined;
	import { append, exitDocument } from 'metal-dom';
	import { getData } from '../data';
	import { render } from './render';

	const patchingComponents_ = [];

	/**
  * Guarantees that the component's element has a parent. That's necessary
  * when calling incremental dom's `patchOuter` for now, as otherwise it will
  * throw an error if the element needs to be replaced.
  * @param {Element} element
  * @return {Element} The parent, in case it was added.
  * @private
  */
	function buildParentIfNecessary_(element) {
		if (!element || !element.parentNode) {
			let parent = {};
			if (typeof document !== 'undefined') {
				parent = document.createElement('div');
			}
			if (element) {
				append(parent, element);
			}
			return parent;
		}
	}

	/**
  * Calls incremental dom's patch function.
  * @param {!Component} component The component to patch.
  * @param {!Element} element The element the component should be patched on.
  * @param {boolean=} outer Flag indicating if `patchOuter` should be used
  *     instead of `patch`.
  * @private
  */
	function callPatch_(component, element, outer) {
		patchingComponents_.push(component);

		const data = getData(component);
		if (!data.render) {
			// Store reference to avoid binds on every patch.
			data.render = render.bind(null, component);
		}

		const patchFn = outer ? IncrementalDOM.patchOuter : IncrementalDOM.patch;
		patchFn(element, data.render);

		patchingComponents_.pop();
	}

	/**
  * Gets the component that triggered the current patch operation.
  * @return {Component}
  */
	export function getPatchingComponent() {
		return patchingComponents_[patchingComponents_.length - 1];
	}

	/**
  * Patches the component with incremental dom function calls.
  * @param {!Component} component
  */
	export function patch(component) {
		if (!tryPatchEmptyWithParent_(component)) {
			if (!tryPatchWithNoParent_(component)) {
				const element = component.element;
				callPatch_(component, element, true);
			}
		}
	}

	/**
  * Checks if the component has no content but was rendered from another
  * component. If so, we'll need to patch this parent to make sure that any new
  * content will be added in the right position.
  * @param {!Component} component
  * @return {?boolean} True if the patch happened. Nothing otherwise.
  * @private
  */
	function tryPatchEmptyWithParent_(component) {
		const data = getData(component);
		if (!component.element && data.parent) {
			data.parent.getRenderer().patch(data.parent);
			return true;
		}
	}

	/**
  * Checks if the component's element exists and has a parent. If that's not the
  * case, a temporary parent will be created and passed to the `patch` function,
  * since incremental dom requires it. Once the patch is done the temporary
  * parent is removed and the component's content is reattached to the correct
  * final position.
  * @param {!Component} component
  * @return {?boolean} True if the patch happened. Nothing otherwise.
  * @private
  */
	function tryPatchWithNoParent_(component) {
		const tempParent = buildParentIfNecessary_(component.element);
		if (tempParent) {
			callPatch_(component, tempParent);
			exitDocument(component.element);
			if (component.element && component.inDocument) {
				const attach = component.getAttachData();
				component.attachElement(attach.parent, attach.sibling);
			}
			return true;
		}
	}
});
//# sourceMappingURL=patch.js.map