package com.hendraanggrian.text;

import android.support.annotation.NonNull;
import android.text.Spannable;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static android.text.Spanned.SPAN_EXCLUSIVE_EXCLUSIVE;

/**
 * Utility class to easily add and remove spans of a Spannable.
 *
 * @author Hendra Anggrian (hendraanggrian@gmail.com)
 */
public final class Spannables {

    private Spannables() {
    }

    /**
     * Set spans from start to end with certain flag.
     */
    public static void setSpans(@NonNull Spannable spannable, int start, int end, @Spannable2.Flags int flags, @NonNull Object... spans) {
        for (Object span : spans) {
            spannable.setSpan(span, start, end, flags);
        }
    }

    /**
     * Set spans from start to end with default flag.
     */
    public static void setSpans(@NonNull Spannable spannable, int start, int end, @NonNull Object... spans) {
        setSpans(spannable, start, end, SPAN_EXCLUSIVE_EXCLUSIVE, spans);
    }

    /**
     * Set spans to the entire text with certain flag.
     */
    public static void setSpans(@NonNull Spannable spannable, @Spannable2.Flags int flags, @NonNull Object... spans) {
        setSpans(spannable, 0, spannable.length(), flags, spans);
    }

    /**
     * Set spans to the entire text with default flag.
     */
    public static void setSpans(@NonNull Spannable spannable, @NonNull Object... spans) {
        setSpans(spannable, 0, spannable.length(), SPAN_EXCLUSIVE_EXCLUSIVE, spans);
    }

    /**
     * Find substring in this Spannable and set multiple spans to it.
     */
    public static void putSpans(@NonNull Spannable spannable, @NonNull CharSequence text, @Spannable2.Flags int flags, @NonNull SpanGetter... getters) {
        String string = spannable.toString();
        String substring = text.toString();
        for (int start : listOccurrences(string, substring)) {
            int end = start + substring.length();
            for (SpanGetter getter : getters) {
                spannable.setSpan(getter.getSpan(), start, end, flags);
            }
        }
    }

    /**
     * Find substring in this Spannable and set multiple spans to it with default flag.
     */
    public static void putSpans(@NonNull Spannable spannable, @NonNull CharSequence text, @NonNull SpanGetter... getters) {
        putSpans(spannable, text, SPAN_EXCLUSIVE_EXCLUSIVE, getters);
    }

    /**
     * Find substring with regex in this Spannable and set multiple spans to it.
     */
    public static void putSpansAll(@NonNull Spannable spannable, @NonNull String regex, @Spannable2.Flags int flags, @NonNull SpanGetter... getters) {
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(spannable);
        while (matcher.find()) {
            int start = matcher.start();
            int end = matcher.end();
            for (SpanGetter getter : getters) {
                spannable.setSpan(getter.getSpan(), start, end, flags);
            }
        }
    }

    /**
     * Find substring with regex in this Spannable and set multiple spans to it with default flag.
     */
    public static void putSpansAll(@NonNull Spannable spannable, @NonNull String regex, @NonNull SpanGetter... getters) {
        putSpansAll(spannable, regex, SPAN_EXCLUSIVE_EXCLUSIVE, getters);
    }

    /**
     * Remove multiple spans in this Spannable.
     */
    public static void removeSpans(@NonNull Spannable spannable, @NonNull Object... spans) {
        for (Object span : spans) {
            spannable.removeSpan(span);
        }
    }

    @NonNull
    static List<Integer> listOccurrences(@NonNull String string, @NonNull String substring) {
        List<Integer> lastIndexes = new ArrayList<>();
        int lastIndex = 0;
        while (lastIndex != -1) {
            lastIndex = string.indexOf(substring, lastIndex);
            if (lastIndex != -1) {
                lastIndexes.add(lastIndex);
                lastIndex += substring.length();
            }
        }
        return lastIndexes;
    }
}