package com.hendraanggrian.text;

import android.support.annotation.NonNull;
import android.support.annotation.RequiresApi;
import android.text.SpannableString;
import android.text.SpannableStringBuilder;

import java.util.List;

/**
 * @author Hendra Anggrian (hendraanggrian@gmail.com)
 */
public final class SpannableText extends SpannableString implements Spannable2 {

    public SpannableText(@NonNull CharSequence text) {
        super(text);
    }

    public SpannableText(@NonNull CharSequence text, @NonNull Object... spans) {
        this(text, SPAN_EXCLUSIVE_EXCLUSIVE, spans);
    }

    public SpannableText(@NonNull CharSequence text, @Flags int flags, @NonNull Object... spans) {
        super(text);
        Spannables.setSpans(this, flags, spans);
    }

    @Override
    public void setSpans(int start, int end, @Flags int flags, @NonNull Object... spans) {
        Spannables.setSpans(this, start, end, flags, spans);
    }

    @Override
    public void setSpans(int start, int end, @NonNull Object... spans) {
        Spannables.setSpans(this, start, end, spans);
    }

    @Override
    public void setSpans(@Flags int flags, @NonNull Object... spans) {
        Spannables.setSpans(this, flags, spans);
    }

    @Override
    public void setSpans(@NonNull Object... spans) {
        Spannables.setSpans(this, spans);
    }

    @Override
    public void removeSpans(@NonNull Object... spans) {
        Spannables.removeSpans(this, spans);
    }

    @Override
    public void putSpans(@NonNull CharSequence text, @Flags int flags, @NonNull SpanGetter... getters) {
        Spannables.putSpans(this, text, flags, getters);
    }

    @Override
    public void putSpans(@NonNull CharSequence text, @NonNull SpanGetter... getters) {
        Spannables.putSpans(this, text, getters);
    }

    @Override
    public void putSpansAll(@NonNull String regex, @Flags int flags, @NonNull SpanGetter... getters) {
        Spannables.putSpansAll(this, regex, flags, getters);
    }

    @Override
    public void putSpansAll(@NonNull String regex, @NonNull SpanGetter... getters) {
        Spannables.putSpansAll(this, regex, getters);
    }

    @NonNull
    public Builder toBuilder() {
        return new Builder(this);
    }

    @NonNull
    public static SpannableText format(@NonNull String format, @NonNull Object... args) {
        List<Integer> list = Spannables.listOccurrences(format, "%");
        for (int start : Spannables.listOccurrences(format, "%%")) {
            list.remove(Integer.valueOf(start));
            list.remove(Integer.valueOf(start + 1));
        }
        if (list.size() != args.length)
            throw new IllegalArgumentException("Expected argument " + list.size() + ", was " + args.length);
        Builder builder = new Builder();
        String remaining = format;
        int lastIndex = 0;
        for (int i = 0; i < args.length; i++) {
            String subformat = format.substring(lastIndex, list.get(i) + 2);
            lastIndex = list.get(i) + 2;
            remaining = remaining.substring(subformat.length());
            builder.append(subformat.substring(0, subformat.length() - 2));
            Object arg = args[i];
            if (arg instanceof SpannedArg)
                builder.append(String.format(subformat.substring(subformat.length() - 2), ((SpannedArg) arg).arg), ((SpannedArg) arg).flags, (Object[]) ((SpannedArg) arg).spans);
            else
                builder.append(String.format(subformat.substring(subformat.length() - 2), arg));
        }
        builder.append(remaining);
        return builder.build();
    }

    @NonNull
    public static SpannableText valueOf(@NonNull CharSequence text) {
        return text instanceof SpannableText
                ? (SpannableText) text
                : new SpannableText(text);
    }

    public static final class Builder extends SpannableStringBuilder implements Spannable2 {

        public Builder() {
            this("");
        }

        public Builder(@NonNull CharSequence text) {
            this(text, 0, text.length());
        }

        public Builder(@NonNull CharSequence text, int start, int end) {
            super(text, start, end);
        }

        @NonNull
        @Override
        public Builder append(char text) {
            super.append(text);
            return this;
        }

        @NonNull
        @Override
        public Builder append(@NonNull CharSequence text) {
            super.append(text);
            return this;
        }

        @NonNull
        @Override
        public Builder append(@NonNull CharSequence text, int start, int end) {
            super.append(text, start, end);
            return this;
        }

        @RequiresApi(21)
        @NonNull
        @Override
        public Builder append(@NonNull CharSequence text, @NonNull Object what, @Flags int flags) {
            super.append(text, what, flags);
            return this;
        }

        @NonNull
        public Builder append(@NonNull CharSequence text, @NonNull Object... spans) {
            return append(text, SPAN_EXCLUSIVE_EXCLUSIVE, spans);
        }

        @NonNull
        public Builder append(@NonNull CharSequence text, @Flags int flags, @NonNull Object... spans) {
            super.append(text);
            int end = length();
            int start = end - text.length();
            for (Object what : spans)
                setSpan(what, start, end, flags);
            return this;
        }

        @Override
        public void setSpans(int start, int end, @Flags int flags, @NonNull Object... spans) {
            Spannables.setSpans(this, start, end, flags, spans);
        }

        @Override
        public void setSpans(int start, int end, @NonNull Object... spans) {
            Spannables.setSpans(this, start, end, spans);
        }

        @Override
        public void setSpans(@Flags int flags, @NonNull Object... spans) {
            Spannables.setSpans(this, flags, spans);
        }

        @Override
        public void setSpans(@NonNull Object... spans) {
            Spannables.setSpans(this, spans);
        }

        @Override
        public void removeSpans(@NonNull Object... spans) {
            Spannables.removeSpans(this, spans);
        }

        @Override
        public void putSpans(@NonNull CharSequence text, @Flags int flags, @NonNull SpanGetter... getters) {
            Spannables.putSpans(this, text, flags, getters);
        }

        @Override
        public void putSpans(@NonNull CharSequence text, @NonNull SpanGetter... getters) {
            Spannables.putSpans(this, text, getters);
        }

        @Override
        public void putSpansAll(@NonNull String regex, @Flags int flags, @NonNull SpanGetter... getters) {
            Spannables.putSpansAll(this, regex, flags, getters);
        }

        @Override
        public void putSpansAll(@NonNull String regex, @NonNull SpanGetter... getters) {
            Spannables.putSpansAll(this, regex, getters);
        }

        @NonNull
        public SpannableText build() {
            return new SpannableText(this);
        }

        @NonNull
        public static Builder valueOf(@NonNull CharSequence text) {
            return text instanceof Builder
                    ? (Builder) text
                    : new Builder(text);
        }
    }
}