package com.gradle.publish;

import org.gradle.api.*;
import org.gradle.api.artifacts.ResolvableDependencies;
import org.gradle.api.artifacts.component.ComponentIdentifier;
import org.gradle.api.artifacts.component.ModuleComponentIdentifier;
import org.gradle.api.artifacts.result.ResolutionResult;
import org.gradle.api.file.SourceDirectorySet;
import org.gradle.api.internal.artifacts.result.DefaultResolvedDependencyResult;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.api.plugins.BasePlugin;
import org.gradle.api.plugins.JavaPlugin;
import org.gradle.api.plugins.JavaPluginConvention;
import org.gradle.api.plugins.JavaPluginExtension;
import org.gradle.api.publish.Publication;
import org.gradle.api.publish.PublishingExtension;
import org.gradle.api.publish.maven.internal.publication.MavenPublicationInternal;
import org.gradle.api.tasks.SourceSet;
import org.gradle.api.tasks.SourceTask;
import org.gradle.api.tasks.TaskProvider;
import org.gradle.api.tasks.bundling.Jar;
import org.gradle.api.tasks.javadoc.Javadoc;
import org.gradle.language.base.plugins.LifecycleBasePlugin;
import org.gradle.plugin.devel.GradlePluginDevelopmentExtension;
import org.gradle.plugin.devel.PluginDeclaration;
import org.gradle.plugins.signing.SigningExtension;
import org.gradle.util.GradleVersion;
import org.gradle.util.VersionNumber;

import javax.annotation.Nonnull;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import static org.codehaus.groovy.runtime.StringGroovyMethods.capitalize;
import static org.gradle.util.GradleVersion.version;

public class PublishPlugin implements Plugin<Project> {
    public static final String JAVA_GRADLE_PLUGIN_ID = "java-gradle-plugin";
    public static final String MAVEN_PUBLISH_PLUGIN_ID = "maven-publish";
    public static final String SHADOW_PLUGIN_ID = "com.github.johnrengelman.shadow";
    public static final String NEW_SHADOW_PLUGIN_ID = "com.gradleup.shadow";
    public static final String SIGNING_PLUGIN_ID = "signing";

    public static final String LOGIN_TASK_NAME = "login";
    private static final String PUBLISH_TASK_DESCRIPTION = "Publishes this plugin to the " +
        "Gradle Plugin portal.";
    private static final String LOGIN_TASK_DESCRIPTION = "Update the gradle.properties files " +
        "so this machine can publish to the Gradle Plugin portal.";
    private static final String PORTAL_BUILD_GROUP_NAME = "Plugin Portal";

    private static final String BASE_TASK_NAME = "publishPlugin";
    private static final String SOURCES_JAR_TASK_NAME = BASE_TASK_NAME + "SourcesJar";
    private static final String JAVA_DOCS_TASK_NAME = BASE_TASK_NAME + "JavaDocsJar";
    private static final String PUBLISH_TASK_NAME = "publishPlugins";
    public static final String PLUGIN_BUNDLE_EXTENSION_NAME = "pluginBundle";

    static final String SOURCES_CLASSIFIER = "sources";
    static final String JAVADOC_CLASSIFIER = "javadoc";

    static final String MAVEN_PUBLICATION_NAME = "pluginMaven";
    static final String MARKER_PUBLICATION_SUFFIX = "PluginMarkerMaven";

    private static final Logger LOGGER = Logging.getLogger(PublishPlugin.class);

    @Override
    public void apply(@Nonnull final Project project) {
        if (runningOnPreGradle(version("4.10"))) {
            throw new RuntimeException("This version of the Plugin Publish Plugin isn't compatible with Gradle versions older than 4.10");
        }

        if (runningOnPreGradle(version("6.0"))) {
            project.getPluginManager().withPlugin(SHADOW_PLUGIN_ID, shadowPlugin -> {
                throw new RuntimeException("This version of the Plugin Publish Plugin isn't compatible with the shadow plugin (" + SHADOW_PLUGIN_ID + "), when running on Gradle versions older than 6.0");
            });
            project.getPluginManager().withPlugin(NEW_SHADOW_PLUGIN_ID, shadowPlugin -> {
                throw new RuntimeException("This version of the Plugin Publish Plugin isn't compatible with the shadow plugin (" + SHADOW_PLUGIN_ID + "), when running on Gradle versions older than 6.0");
            });
        }

        project.getPlugins().apply("org.gradle.java-library");

        final PluginBundleExtension bundle;
        if (runningOnPreGradle(version("8.0"))) {
            bundle = new PluginBundleExtension();
            project.getExtensions().add(PLUGIN_BUNDLE_EXTENSION_NAME, bundle);
        } else {
            bundle = null;
        }

        Log4jVulnerabilityChecker.decorate(project);

        project.getTasks().register(PUBLISH_TASK_NAME, PublishTask.class, publishTask -> {
            publishTask.setDescription(PUBLISH_TASK_DESCRIPTION);
            publishTask.setGroup(PORTAL_BUILD_GROUP_NAME);
            publishTask.setBundleConfig(bundle);
            markAsNotCompatibleWithConfigurationCache(publishTask);
        });

        LOGGER.debug("Setup: " + PUBLISH_TASK_NAME + " of " + getClass().getName());

        project.getTasks().register(LOGIN_TASK_NAME, LoginTask.class, loginTask -> {
            loginTask.setDescription(LOGIN_TASK_DESCRIPTION);
            loginTask.setGroup(PORTAL_BUILD_GROUP_NAME);
        });

        LOGGER.debug("Created task: " + LOGIN_TASK_NAME + " of " + getClass().getName());

        wireToJavaGradlePluginAndMavenPublish(project);

        project.afterEvaluate(finalProject -> {
            final GradlePluginDevelopmentExtension pluginDevelopmentExtension = project.getExtensions().getByType(GradlePluginDevelopmentExtension.class);
            if (!pluginDevelopmentExtension.isAutomatedPublishing()) {
                throw new RuntimeException("Since version 1.0 of the Plugin Publish plugin non-automatic publishing is not allowed anymore; the Maven Publish plugin must be used to generate publication metadata");
            }

            List<Task> signingTasks = enableSigning(project);

            PublishTask publishTask = (PublishTask) project.getTasks().findByName(PUBLISH_TASK_NAME);
            if (publishTask != null) {
                publishTask.dependOnPublishTasks();
                publishTask.dependsOn(LifecycleBasePlugin.ASSEMBLE_TASK_NAME);
                signingTasks.forEach(publishTask::dependsOn);
            }

            project.getPluginManager().withPlugin(PublishPlugin.SHADOW_PLUGIN_ID, new PublishTaskShadowAction(project, LOGGER));
            project.getPluginManager().withPlugin(PublishPlugin.NEW_SHADOW_PLUGIN_ID, new PublishTaskShadowAction(project, LOGGER));

            forceJavadocAndSourcesJars(project);
        });
    }

    private List<Task> enableSigning(Project project) {
        List<Task> signingTasks = new ArrayList<>();
        project.getPluginManager().withPlugin(PublishPlugin.SIGNING_PLUGIN_ID, signingPlugin -> {
            LOGGER.info("Signing plugin detected. Will automatically sign the published artifacts.");

            SigningExtension signing = project.getExtensions().getByType(SigningExtension.class);
            PublishingExtension publishing = project.getExtensions().getByType(PublishingExtension.class);
            Task mavenPublishSignTask = getSigningTask(project, signing, publishing.getPublications().getByName(MAVEN_PUBLICATION_NAME));
            if (mavenPublishSignTask != null) {
                signingTasks.add(mavenPublishSignTask);
            }

            NamedDomainObjectContainer<PluginDeclaration> plugins = project.getExtensions().getByType(GradlePluginDevelopmentExtension.class).getPlugins();
            for (PluginDeclaration plugin : plugins) {
                String markerPublicationName = plugin.getName() + MARKER_PUBLICATION_SUFFIX;
                Publication publication = publishing.getPublications().getByName(markerPublicationName);
                Task pluginSigningTask = getSigningTask(project, signing, publication);
                if (pluginSigningTask != null) {
                    signingTasks.add(pluginSigningTask);
                }
            }
        });
        return signingTasks;
    }

    private Task getSigningTask(Project project, SigningExtension signing, Publication publication) {
        String signTaskName = determineSignTaskNameForPublication(publication);
        Task signTask = project.getTasks().findByName(signTaskName);
        if (signTask == null) {
            signTask = signing.sign(publication).get(0);
        }
        return signTask;
    }

    private static String determineSignTaskNameForPublication(Publication publication) {
        return "sign" + capitalize((CharSequence) publication.getName()) + "Publication";
    }

    private void markAsNotCompatibleWithConfigurationCache(PublishTask publishTask) {
        if (!runningOnPreGradle(version("7.4"))) {
            try {
                Method method = Task.class.getMethod("notCompatibleWithConfigurationCache", String.class);
                method.invoke(publishTask, "Plugin Publish plugin not yet compatible with Configuration Cache.");
            } catch (Exception e) {
                throw new RuntimeException("Failed marking the " + BASE_TASK_NAME + " as not compatible with Configuration Cache", e);
            }
        }
    }

    private void forceJavadocAndSourcesJars(Project project) {
        if (runningOnPreGradle(version("6.0"))) {
            registerExtraArtifactProducingTask(project, SOURCES_JAR_TASK_NAME, createAndSetupJarSourcesTask(project));
            registerExtraArtifactProducingTask(project, JAVA_DOCS_TASK_NAME, createAndSetupJavaDocsTask(project));
        } else {
            JavaPluginExtension javaPluginExtension = project.getExtensions().getByType(JavaPluginExtension.class);
            javaPluginExtension.withJavadocJar();
            javaPluginExtension.withSourcesJar();
        }
    }

    private void registerExtraArtifactProducingTask(Project project, String taskName, TaskProvider<Jar> taskProvider) {
        Task task = project.getTasks().findByName(taskName);
        if (task == null) {
            task = taskProvider.get();
        }

        Task assembleTask = project.getTasks().findByName(LifecycleBasePlugin.ASSEMBLE_TASK_NAME);
        assembleTask.dependsOn(task);

        PublishingExtension publishing = project.getExtensions().getByType(PublishingExtension.class);
        MavenPublicationInternal publication = (MavenPublicationInternal) publishing.getPublications().getByName(PublishTask.MAVEN_PUBLICATION_NAME);
        if (!publication.getArtifacts().contains(task)) {
            publication.getArtifacts().artifact(task);
        }
    }

    private TaskProvider<Jar> createAndSetupJavaDocsTask(Project project) {
        return createBasicDocJarTask(project, JAVA_DOCS_TASK_NAME, JAVADOC_CLASSIFIER, "Assembles a jar archive containing the documentation for the main Java source code.",
            docsTask -> {
                Javadoc javadoc = (Javadoc) project.getTasks().findByName(JavaPlugin.JAVADOC_TASK_NAME);
                docsTask.from(javadoc.getDestinationDir());
            });
    }

    private TaskProvider<Jar> createBasicDocJarTask(Project project, String name, String classifier, String description, Action<? super Jar> extraConfigAction) {
        return project.getTasks().register(name, Jar.class, docsTask -> {
            docsTask.setDescription(description);
            docsTask.setGroup(BasePlugin.BUILD_GROUP);
            setClassifier(docsTask, classifier);
            extraConfigAction.execute(docsTask);
        });
    }

    private TaskProvider<Jar> createAndSetupJarSourcesTask(Project project) {
        return project.getTasks().register(SOURCES_JAR_TASK_NAME, Jar.class, sourcesTask -> {
            sourcesTask.setDescription("Assembles a jar archive containing the main source code.");
            sourcesTask.setGroup(BasePlugin.BUILD_GROUP);
            setClassifier(sourcesTask, SOURCES_CLASSIFIER);
            JavaPluginConvention javaPluginConvention = project.getConvention().findPlugin(JavaPluginConvention.class);
            SourceSet mainSourceSet = javaPluginConvention.getSourceSets().getByName(SourceSet.MAIN_SOURCE_SET_NAME);
            SourceDirectorySet allMainSources = mainSourceSet.getAllSource();
            sourcesTask.from(allMainSources);
        });
    }

    private void setClassifier(Jar task, String classifier){
        if (runningOnPreGradle(version("7.0"))) {
            try {
                Jar.class.getMethod("setClassifier", String.class).invoke(task, classifier);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        } else {
            task.getArchiveClassifier().set(classifier);
        }
    }

    private void wireToJavaGradlePluginAndMavenPublish(final Project project) {
        project.getPluginManager().apply(JAVA_GRADLE_PLUGIN_ID);
        project.getPluginManager().apply(MAVEN_PUBLISH_PLUGIN_ID);
    }

    private static boolean runningOnPreGradle(GradleVersion version) {
        return GradleVersion.current().getBaseVersion().compareTo(version) < 0;
    }

    private static class Log4jVulnerabilityChecker {

        public static final String ERROR_MESSAGE = "Cannot publish a plugin which resolves a vulnerable Log4j version (https://blog.gradle.org/log4j-vulnerability). " +
            "Make sure to update your configuration so it does not happen.";

        static void decorate(Project project) {
            project.getConfigurations().all(conf -> {
                ResolvableDependencies incoming = conf.getIncoming();
                incoming.afterResolve(rd -> {
                    ResolutionResult resolutionResult = rd.getResolutionResult();
                    resolutionResult.allDependencies(dependencyResult -> {
                        if (dependencyResult instanceof DefaultResolvedDependencyResult) {
                            ComponentIdentifier componentIdentifier = ((DefaultResolvedDependencyResult) dependencyResult).getSelected().getId();
                            if (componentIdentifier instanceof ModuleComponentIdentifier) {
                                ModuleComponentIdentifier moduleComponentIdentifier = (ModuleComponentIdentifier) componentIdentifier;
                                if (isVulnerableLog4jDependency(moduleComponentIdentifier)) {
                                    throw new GradleException(ERROR_MESSAGE);
                                }
                            }
                        }
                    });
                });
            });
        }

        private static boolean isVulnerableLog4jDependency(ModuleComponentIdentifier identifier) {
            if (!"org.apache.logging.log4j".equals(identifier.getGroup())) {
                return false;
            }

            if (!"log4j-core".equals(identifier.getModule())) {
                return false;
            }

            VersionNumber versionNumber = VersionNumber.parse(identifier.getVersion());
            return versionNumber.compareTo(new VersionNumber(2, 0, 0, "alpha1")) >= 0 &&
                versionNumber.compareTo(new VersionNumber(2, 17, 1, null)) < 0;
        }
    }
}
