package com.gradle.publish;

import com.gradle.protocols.ServerResponseBase;
import com.gradle.publish.protocols.v1.models.ClientPostRequest;
import com.gradle.publish.protocols.v1.models.publish.PublishActivateRequest;
import com.gradle.publish.protocols.v1.models.publish.PublishArtifact;
import com.gradle.publish.protocols.v1.models.publish.PublishMavenCoordinates;
import com.gradle.publish.protocols.v1.models.publish.PublishNewVersionRequest;
import com.gradle.publish.protocols.v1.models.publish.PublishNewVersionResponse;
import com.gradle.publish.protocols.v1.models.publish.ValidateNewVersionRequest;
import com.gradle.publish.protocols.v1.models.publish.ValidateNewVersionResponse;
import com.gradle.publish.upload.Uploader;
import org.gradle.api.Project;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import static com.gradle.publish.PublishTask.GRADLE_PUBLISH_KEY;
import static com.gradle.publish.PublishTask.GRADLE_PUBLISH_KEY_ENV;
import static com.gradle.publish.PublishTask.GRADLE_PUBLISH_SECRET;
import static com.gradle.publish.PublishTask.GRADLE_PUBLISH_SECRET_ENV;


class PortalPublisher {

    private static final Logger LOGGER = Logging.getLogger(PublishTask.class);
    private final Config ghConfig;
    private final Project project;

    PortalPublisher(Project project) {
        this.project = project;
        this.ghConfig = new Config(project);
    }

    void publishToPortal(
        List<PublishNewVersionRequest> requests,
        PublishMavenCoordinates mavenCoords,
        Map<PublishArtifact, File> artifacts
    ) throws Exception {

        List<PublishNewVersionResponse> apiResponses = new ArrayList<>();

        // do all handshakes first to make sure all are in order
        for (PublishNewVersionRequest request : requests) {
            LOGGER.info("Publishing plugin {} version {}", request.getPluginId(), request.getPluginVersion());

            // attach maven coords and artifacts
            request.setMavenCoordinates(mavenCoords);
            request.setArtifacts(new ArrayList<>(artifacts.keySet()));

            PublishNewVersionResponse apiResponse = doSignedPost(request);
            handleApiResponse(request.getPluginId(), apiResponse);
            apiResponses.add(apiResponse);
        }

        // publish the artifacts, but only once
        LOGGER.debug("Uploading artifacts");
        publishArtifacts(apiResponses.get(0), artifacts);

        // now activate plugins
        for (PublishNewVersionResponse apiResponse : apiResponses) {
            activate(apiResponse.getNextRequest());
        }
    }

    void validatePublishingToPortal(
        List<ValidateNewVersionRequest> requests,
        PublishMavenCoordinates mavenCoords,
        Map<PublishArtifact, File> artifacts
    ) throws Exception {
        for (ValidateNewVersionRequest request : requests) {
            LOGGER.info("Validating plugin publication of {} version {}", request.getPluginId(), request.getPluginVersion());

            // attach maven coords and artifacts
            request.setMavenCoordinates(mavenCoords);
            request.setArtifacts(new ArrayList<>(artifacts.keySet()));

            ValidateNewVersionResponse apiResponse = doSignedPost(request);
            handleValidationResponse(request.getPluginId(), apiResponse);
        }
    }

    private <T extends ServerResponseBase> T doSignedPost(ClientPostRequest<T> postRequest)
        throws Exception {
        return buildOAuthClient().send(postRequest);
    }

    private OAuthHttpClient buildOAuthClient() {
        Properties props = createPropertiesStore();
        String key = getEnvThenSystemProperty(GRADLE_PUBLISH_KEY_ENV, props, GRADLE_PUBLISH_KEY);
        String secret = getEnvThenSystemProperty(GRADLE_PUBLISH_SECRET_ENV, props, GRADLE_PUBLISH_SECRET);
        if (key == null || key.trim().isEmpty() || secret == null || secret.trim().isEmpty()) {
            throw new IllegalArgumentException("Missing publishing keys. Please set " +
                GRADLE_PUBLISH_KEY + "/" + GRADLE_PUBLISH_SECRET +
                " system properties or " + GRADLE_PUBLISH_KEY_ENV + "/" + GRADLE_PUBLISH_SECRET_ENV +
                " env variables or login using the " + PublishPlugin.LOGIN_TASK_NAME +
                " task.");
        }
        return new OAuthHttpClient(ghConfig.getPortalUrl(), key, secret);
    }

    private Properties createPropertiesStore() {
        return PropertiesStore.all(project);
    }

    private String getEnvThenSystemProperty(String envName, Properties properties, String propertyName) {
        String envValue = System.getenv(envName);
        if (envValue == null) {
            return properties.getProperty(propertyName);
        } else {
            return envValue;
        }
    }

    private void handleApiResponse(String pluginId, PublishNewVersionResponse apiResponse) {
        ResponseUtil.assertValidResponse("Request to publish new plugin '" + pluginId + "' failed!", apiResponse);
        if (apiResponse.hasFailed()) {
            throw new RuntimeException("Cannot publish plugin '" + pluginId + "'\n"
                + "Server responded with: " + apiResponse.getErrorMessage());
        }
        if (apiResponse.hasWarning()) {
            LOGGER.warn(apiResponse.getWarningMessage());
        }
    }

    private void handleValidationResponse(String pluginId, ValidateNewVersionResponse apiResponse) {
        ResponseUtil.assertValidResponse("Request to publish new plugin '" + pluginId + "' failed!", apiResponse);
        if (apiResponse.hasFailed()) {
            throw new RuntimeException("Cannot publish plugin '" + pluginId + "'\n"
                + "Server responded with: " + apiResponse.getErrorMessage());
        }
    }

    private void publishArtifacts(PublishNewVersionResponse apiResponse,
                                  Map<PublishArtifact, File> artifactHashes) throws IOException {
        Map<String, String> publishedHashAndUrls = apiResponse.getPublishTo();
        for (Map.Entry<PublishArtifact, File> art : artifactHashes.entrySet()) {
            String uploadUrl = publishedHashAndUrls.get(art.getKey().encode());
            File artifactFile = art.getValue();
            uploadArtifactIfNecessary(artifactFile, uploadUrl);
        }
    }

    private void uploadArtifactIfNecessary(File artifactFile, String uploadUrl) throws IOException {
        URI filePath = project.getProjectDir().toURI().relativize(artifactFile.toURI());
        if (uploadUrl != null) {
            LOGGER.info("Publishing artifact {}", filePath);
            LOGGER.debug("Publishing {} to {}", filePath, uploadUrl);
            Uploader.putFile(artifactFile, uploadUrl);
        } else {
            LOGGER.info("Skipping upload of artifact {} as it has been previously uploaded", filePath);
        }
    }

    private void activate(PublishActivateRequest activePlugin) throws Exception {
        LOGGER.info("Activating plugin {} version {}", activePlugin.getPluginId(),
            activePlugin.getVersion());
        doSignedPost(activePlugin);
    }
}
