package com.gradle.publish;

import org.gradle.plugin.devel.GradlePluginDevelopmentExtension;
import org.gradle.plugin.devel.PluginDeclaration;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

import static com.gradle.publish.Util.isBlank;
import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableList;

abstract class AbstractConfigValidator {

    protected static final Collection<String> FORBIDDEN_NAMESPACES =
            unmodifiableList(asList("org.gradle", "com.gradleware", "com.gradle", "org.samples.greeting"));

    // taken directly from http://svn.apache.org/repos/asf/maven/maven-2/branches/maven-2.2.x/maven-project/src/main/java/org/apache/maven/project/validation/DefaultModelValidator.java
    // thanks StackOverflow! http://stackoverflow.com/a/2112506
    private static final Pattern MAVEN_ID_REGEX = Pattern.compile("[A-Za-z0-9_\\-.]+");

    private final boolean skipNamespaceCheck;

    AbstractConfigValidator(boolean skipNamespaceCheck) {
        this.skipNamespaceCheck = skipNamespaceCheck;
    }

    public abstract void validateConfig(GradlePluginDevelopmentExtension pluginConfig);

    public abstract void validatePluginConfig(PluginDeclaration plugin);

    public abstract void validateTags(PluginDeclaration plugin);

    public void validateMavenCoordinates(String groupId, String artifactId, String version) {
        if (isBlank(groupId)) {
            throw new IllegalArgumentException("Cannot publish artifacts as no group ID set.\n"
                    + "Please set the group at the project level.");
        }
        if (!MAVEN_ID_REGEX.matcher(groupId).matches()) {
            throw new IllegalArgumentException(
                    "Cannot publish artifacts as group ID '" + groupId + "' is invalid.\n"
                            + "Valid group IDs may include alphanumeric characters, numbers, dashes,\n"
                            + "underscores and dots.");
        }
        if (failsNamespaceCheck(groupId)) {
            throw new IllegalArgumentException(
                    "Cannot publish artifacts as group ID '" + groupId + "' is not allowed.\n"
                            + "Group IDs cannot begin with " + String.join(", ", FORBIDDEN_NAMESPACES));
        }

        if (!MAVEN_ID_REGEX.matcher(artifactId).matches()) {
            throw new IllegalArgumentException(
                    "Cannot publish artifacts as artifact ID '" + artifactId + "' is invalid.\n"
                            + "Valid artifact IDs may include alphanumeric characters, numbers, dashes,\n"
                            + "underscores and dots.");
        }

        if (artifactId.equals(groupId + ".gradle.plugin")) {
            throw new IllegalArgumentException("Group ID + '.gradle.plugin' suffix not allowed as artifact ID");
        }

        if (isBlank(version) || version.equals("unspecified")) {
            throw new IllegalArgumentException("Cannot publish artifacts as no version set.\n"
                    + "Please set the version at the project level.");

        }
    }

    protected void validateURLs(String websiteUrl, String vcsUrl) {
        validateURL(websiteUrl, "Website");
        validateURL(vcsUrl, "VCS");
    }

    private void validateURL(String url, String description) {
        if (isBlank(url)) {
            throw new IllegalArgumentException(description + " URL not set");
        }
        try {
            URI normalizedURI = new URI(url.toLowerCase());
            String scheme = normalizedURI.getScheme();
            if (!scheme.equals("http") && !scheme.equals("https")) {
                throw new IllegalArgumentException(description + " URL only supports HTTP or HTTPS schemas");
            }
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Can't parse URL '" + url + "'");
        }
    }

    protected void validatePluginIdUniqueness(GradlePluginDevelopmentExtension pluginConfig) {
        Set<String> usedPluginIds = new HashSet<>();
        for(PluginDeclaration plugin : pluginConfig.getPlugins()) {
            String pluginId = plugin.getId();
            if (usedPluginIds.contains(pluginId)) {
                throw new InvalidPluginIdException(pluginId, "multiple plugins are using the same ID");
            }
            usedPluginIds.add(pluginId);
        }
    }

    protected void validatePluginId(String pluginName, String pluginId) {
        if (pluginId == null || pluginId.isEmpty()) {
            throw new IllegalArgumentException("Plugin " + pluginName + " does not define a plugin ID. Please set the 'id' property in the plugin declaration");
        }
        PluginId id = PluginId.of(pluginId);
        if (!id.isQualified()) {
            throw new InvalidPluginIdException(pluginId, "Plugin IDs should be namespaced, e.g. 'com.example." + pluginId + "'");
        }
        if (failsNamespaceCheck(pluginId)) {
            throw new InvalidPluginIdException(pluginId, "cannot begin with " + FORBIDDEN_NAMESPACES);
        }
    }

    protected void validateTags(String pluginName, Collection<String> tags) {
       if (tags == null || tags.isEmpty()) {
           throw new IllegalArgumentException("Plugin '" + pluginName + "' has no 'tags' property set");
       }
    }

    protected void validateDisplayName(String pluginName, String displayName) {
        validateProperty(pluginName, "displayName", displayName);
    }

    protected void validateDescription(String pluginName, String description) {
        validateProperty(pluginName, "description", description);
    }

    protected void validateImplementationClass(String pluginName, String implementationClass) {
        validateProperty(pluginName, "implementationClass", implementationClass);
    }

    private void validateProperty(String pluginName, String propertyName, String propertyValue) {
        if (propertyValue == null || isBlank(propertyValue)) {
            throw new IllegalArgumentException("Plugin '" + pluginName + "' has no '" + propertyName + "' property set");
        }
    }

    private boolean failsNamespaceCheck(String id) {
        if (!skipNamespaceCheck) {
            id = id.toLowerCase();
            for(String ns : FORBIDDEN_NAMESPACES) {
                if (id.equals(ns) || id.startsWith(ns + ".")) {
                    return true;
                }
            }
        }
        return false;
    }
}
