/*
 * Copyright (c) 2006-2013 Rogério Liesenfeld
 * This file is subject to the terms of the MIT license (see LICENSE.txt).
 */
package mockit.coverage;

import org.jetbrains.annotations.*;

import mockit.coverage.data.*;
import mockit.coverage.lines.*;

@SuppressWarnings("unused")
public final class TestRun
{
   /**
    * Used to prevent reentrancy in methods which gather coverage information, when measuring the coverage of
    * JMockit Coverage itself.
    */
   @NotNull private static final ThreadLocal<Boolean> executingCall = new ThreadLocal<Boolean>() {
      @Override
      protected Boolean initialValue() { return false; }
   };

   private TestRun() {}

   public static void lineExecuted(int fileIndex, int line)
   {
      if (executingCall.get()) {
         return;
      }

      executingCall.set(true);

      CoverageData coverageData = CoverageData.instance();
      PerFileLineCoverage fileData = coverageData.getFileData(fileIndex).lineCoverageInfo;
      CallPoint callPoint = null;

      if (coverageData.isWithCallPoints() && fileData.acceptsAdditionalCallPoints(line)) {
         callPoint = CallPoint.create(new Throwable());
      }

      fileData.registerExecution(line, callPoint);

      executingCall.set(false);
   }

   public static void jumpTargetExecuted(int fileIndex, int line, int segment)
   {
      if (executingCall.get()) {
         return;
      }

      executingCall.set(true);

      CoverageData coverageData = CoverageData.instance();
      PerFileLineCoverage fileData = coverageData.getFileData(fileIndex).lineCoverageInfo;
      CallPoint callPoint = null;

      if (coverageData.isWithCallPoints() && fileData.acceptsAdditionalCallPoints(line, segment)) {
         callPoint = CallPoint.create(new Throwable());
      }

      fileData.registerExecution(line, segment, true, callPoint);

      executingCall.set(false);
   }

   public static void noJumpTargetExecuted(int fileIndex, int line, int segment)
   {
      if (executingCall.get()) {
         return;
      }

      executingCall.set(true);

      CoverageData coverageData = CoverageData.instance();
      PerFileLineCoverage fileData = coverageData.getFileData(fileIndex).lineCoverageInfo;
      CallPoint callPoint = null;

      if (coverageData.isWithCallPoints() && fileData.acceptsAdditionalCallPoints(line, segment)) {
         callPoint = CallPoint.create(new Throwable());
      }

      fileData.registerExecution(line, segment, false, callPoint);

      executingCall.set(false);
   }

   public static void nodeReached(@NotNull String file, int firstLineInMethodBody, int node)
   {
      if (executingCall.get()) {
         return;
      }

      executingCall.set(true);

      CoverageData coverageData = CoverageData.instance();
      FileCoverageData fileData = coverageData.getFileData(file);
      fileData.pathCoverageInfo.registerExecution(firstLineInMethodBody, node);

      executingCall.set(false);
   }

   public static void fieldAssigned(@NotNull String file, @NotNull String classAndFieldNames)
   {
      CoverageData coverageData = CoverageData.instance();
      FileCoverageData fileData = coverageData.getFileData(file);
      fileData.dataCoverageInfo.registerAssignmentToStaticField(classAndFieldNames);
   }

   public static void fieldRead(@NotNull String file, @NotNull String classAndFieldNames)
   {
      CoverageData coverageData = CoverageData.instance();
      FileCoverageData fileData = coverageData.getFileData(file);
      fileData.dataCoverageInfo.registerReadOfStaticField(classAndFieldNames);
   }

   public static void fieldAssigned(@NotNull Object instance, @NotNull String file, @NotNull String classAndFieldNames)
   {
      CoverageData coverageData = CoverageData.instance();
      FileCoverageData fileData = coverageData.getFileData(file);
      fileData.dataCoverageInfo.registerAssignmentToInstanceField(instance, classAndFieldNames);
   }

   public static void fieldRead(@NotNull Object instance, @NotNull String file, @NotNull String classAndFieldNames)
   {
      CoverageData coverageData = CoverageData.instance();
      FileCoverageData fileData = coverageData.getFileData(file);
      fileData.dataCoverageInfo.registerReadOfInstanceField(instance, classAndFieldNames);
   }
}
