package com.ark.dict;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.SharedPreferences;
import android.os.Build;
import android.preference.PreferenceManager;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.util.*;

public class SharedPrefUtils {

    private static List<Class<?>> CLASSES = new ArrayList<>();
    private static SharedPreferences prefs; // cache

    static {
        CLASSES.add(String.class);
        CLASSES.add(Boolean.class);
        CLASSES.add(Integer.class);
        CLASSES.add(Long.class);
        CLASSES.add(Float.class);
        CLASSES.add(Set.class);
    }

    private SharedPrefUtils() {
    }

    public static SharedPreferences getPrefs() {
        // synchronized is really needed or volatile is all I need (visibility)
        // the same instance of SharedPreferences will be returned AFAIC
        SharedPreferences result = prefs;
        if (result == null)
            synchronized (SharedPrefUtils.class) {
                result = prefs;
                if (result == null) {
                    result = prefs = PreferenceManager
                            .getDefaultSharedPreferences(Utils.getContext());
                }
            }
        return result;
    }

    @TargetApi(Build.VERSION_CODES.GINGERBREAD)
    public static <T> void put(String key, T value) {
        SharedPreferences.Editor ed = _put(key, value);
        ed.apply();
    }

    public static <T> boolean commit(String key, T value) {
        return _put(key, value).commit();
    }

    @SuppressLint("CommitPrefEdits")
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    private static <T> SharedPreferences.Editor _put(String key, T value) {
        if (key == null) {
            throw new NullPointerException("Null keys are not permitted");
        }
        SharedPreferences.Editor ed = getPrefs().edit();
        if (value == null) {
            // commit it as that is exactly what the API does (but not for boxed
            // primitives) - can be retrieved as anything but if you give get()
            // a default non null value it will give this default value back
            ed.putString(key, null);
            // btw the signature is given by the compiler as :
            // <Object> void
            // gr.uoa.di.android.helpers.AccessPreferences.put(Context ctx,
            // String key, Object value)
            // if I write AccessPreferences.put(ctx, "some_key", null);
        } else if (value instanceof String) {
            ed.putString(key, (String) value);
        } else if (value instanceof Boolean) {
            ed.putBoolean(key, (Boolean) value);
        } else if (value instanceof Integer) {
            // while int "is-a" long (will be converted to long) Integer IS NOT
            // a
            // Long (CCE) - so the order of "instanceof" checks does not matter
            // -
            // except for frequency I use the values (so I put String, Boolean
            // and
            // Integer first as I mostly use those preferences)
            ed.putInt(key, (Integer) value);
        } else if (value instanceof Long) {
            ed.putLong(key, (Long) value);
        } else if (value instanceof Float) {
            ed.putFloat(key, (Float) value);
        } else if (value instanceof Set) {
            @SuppressWarnings({"unchecked", "unused"})
            // this set can contain whatever it wants - don't be fooled by the
                    // Set<String> cast
                    SharedPreferences.Editor dummyVariable = ed.putStringSet(key, (Set<String>) value);
        } else {
            throw new IllegalArgumentException("The given value : " + value
                    + " cannot be persisted");
        }
        return ed;
    }

    @SuppressWarnings("unchecked")
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    public static <T> T get(String key) {
        return get(key, null);
    }

    @SuppressWarnings("unchecked")
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    public static <T> T get(String key, T defaultValue) {
        if (key == null) {
            throw new NullPointerException("Null keys are not permitted");
        }
        SharedPreferences preferences = getPrefs();
        if (defaultValue == null) {
            if (!preferences.contains(key))
                return null;
            // if the key does exist I get the value and..
            Object value = preferences.getAll().get(key);
            // ..if null I return null - here I differ from framework - I return
            // null for boxed primitives
            if (value == null)
                return null;
            // ..if not null I get the class of the non null value. Here I
            // differ from framework - I do not throw if the (non null) value is
            // not of the type the variable to receive it is - cause I have no
            // way to guess the return value expected ! (***)
            Class<?> valueClass = value.getClass();
            // the order of "instanceof" checks does not matter - still if I
            // have a long autoboxed as Integer ? - tested in
            // testAPNullDefaultUnboxingLong() and works OK (long 0L is
            // autoboxed as long)
            for (Class<?> cls : CLASSES) {
                if (valueClass.isAssignableFrom(cls)) {
                    return (T) valueClass.cast(value);
                }
            }
            // that's really Illegal State I guess
            throw new IllegalStateException("Unknown class for value :\n\t"
                    + value + "\nstored in preferences");
        } else if (defaultValue instanceof String) {// the order should not
            // matter
            return (T) preferences.getString(key, (String) defaultValue);
        } else if (defaultValue instanceof Boolean) {
            return (T) (Boolean) preferences.getBoolean(key,
                    (Boolean) defaultValue);
        } else if (defaultValue instanceof Integer) {
            return (T) (Integer) preferences.getInt(key,
                    (Integer) defaultValue);
        } else if (defaultValue instanceof Long) {
            return (T) (Long) preferences.getLong(key, (Long) defaultValue);
        } else if (defaultValue instanceof Float) {
            return (T) (Float) preferences
                    .getFloat(key, (Float) defaultValue);
        } else if (defaultValue instanceof Set) {
            return (T) preferences.getStringSet(key,
                    (Set<String>) defaultValue);
        } else {
            throw new IllegalArgumentException(defaultValue
                    + " cannot be persisted in SharedPreferences");
        }
    }

    public static boolean contains(String key) {
        if (key == null) {
            throw new NullPointerException("Null keys are not permitted");
        }
        return getPrefs().contains(key);
    }

    public static Map<String, ?> getAll() {
        return Collections.unmodifiableMap(getPrefs().getAll());
    }

    public static boolean clear() {
        return getPrefs().edit().clear().commit();
    }

    public static boolean remove(String key) {
        if (key == null) {
            throw new NullPointerException("Null keys are not permitted");
        }
        return getPrefs().edit().remove(key).commit();
    }

    public static void registerListener(
            SharedPreferences.OnSharedPreferenceChangeListener lis) {
        if (lis == null) {
            throw new NullPointerException("Null listener");
        }
        getPrefs().registerOnSharedPreferenceChangeListener(lis);
    }

    public static void unregisterListener(
            SharedPreferences.OnSharedPreferenceChangeListener lis) {
        if (lis == null) {
            throw new NullPointerException("Null listener");
        }
        getPrefs().unregisterOnSharedPreferenceChangeListener(lis);
    }

    public static void callListener(
            SharedPreferences.OnSharedPreferenceChangeListener lis, String key) {
        if (lis == null) {
            throw new NullPointerException("Null listener");
        }
        if (key == null) {
            throw new NullPointerException("Null keys are not permitted");
        }
        lis.onSharedPreferenceChanged(getPrefs(), key);
    }

    @SuppressWarnings("unused")
    private static Set<String> checkSetContainsStrings(Set<?> set) {
        if (!set.isEmpty()) {
            for (Object object : set) {
                if (!(object instanceof String)) {
                    throw new IllegalArgumentException(
                            "The given set does not contain strings only");
                }
            }
        }
        @SuppressWarnings("unchecked")
        Set<String> stringSet = (Set<String>) set;
        return stringSet;
    }

    public static <T> void setDataList(String tag, List<T> dataList) {
        if (null == dataList || dataList.size() <= 0)
            return;
        Gson gson = new Gson();
        //转换成json数据，再保存
        String strJson = gson.toJson(dataList);
        put(tag, strJson);
    }

    public static <T> List<T> getDataList(String tag) {
        List<T> dataList = new ArrayList<>();
        String strJson = prefs.getString(tag, null);
        if (null == strJson) {
            return dataList;
        }
        Gson gson = new Gson();
        dataList = gson.fromJson(strJson, new TypeToken<List<T>>() {
        }.getType());
        return dataList;
    }
}
